import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Skeleton for opening a drawer by rotating, moving, picking and pulling.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset to initial state
        descriptions, obs = task.reset()

        # Initialize video recording
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object/world positions
        positions = get_object_positions()
        # We expect keys like 'drawer_side', 'drawer_anchor', 'drawer_handle'
        drawer_side_pos = positions['drawer_side']
        drawer_anchor_pos = positions['drawer_anchor']
        drawer_handle_pos = positions['drawer_handle']

        # 1) Rotate gripper to 90 degrees around the correct axis
        print("[Task] Rotating gripper to 90 degrees")
        # quaternion for 90° around, e.g., z axis
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        obs, reward, done = rotate(env, task, ninety_deg_quat,
                                   max_steps=100, threshold=0.02, timeout=10.0)
        if done:
            print("[Task] Episode ended during rotate")
            return

        # 2) Move to side position of drawer
        print("[Task] Moving to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_side_pos,
                                 approach_distance=0.10,
                                 max_steps=100,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Task] Episode ended during move-to-side")
            return

        # 3) Move to anchor position (handle) of drawer
        print("[Task] Moving to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.05,
                                 max_steps=100,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Task] Episode ended during move-to-anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Grasping drawer handle at:", drawer_anchor_pos)
        obs, reward, done = pick(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.02,
                                 max_steps=100,
                                 threshold=0.005,
                                 timeout=10.0)
        if done:
            print("[Task] Episode ended during pick-drawer")
            return

        # 5) Attempt to pull the drawer and identify lock status
        print("[Task] Attempting to pull the drawer")
        lock_known = False
        try:
            obs, reward, done = pull(env, task,
                                     max_steps=50,
                                     timeout=5.0)
            lock_known = True
            print("[Task] Pull succeeded: drawer is not locked (lock-known=True)")
        except Exception as e:
            # Assume pull raised because drawer was locked
            lock_known = False
            print(f"[Task] Pull failed, lock status unknown (lock-known=False). Exception: {e}")

        print(f"[Task] Final lock_known flag: {lock_known}")

    finally:
        # Ensure the environment always shuts down
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()