import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # get all of the object and position information
        positions = get_object_positions()
        # assume get_object_positions returns keys:
        #   'drawer_side_pos', 'drawer_anchor_pos'
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            raise KeyError("Missing required position keys in get_object_positions()")

        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']

        # 1) Rotate gripper to 90 degrees about the drawer handle
        #    we assume a helper or a fixed quaternion for ninety degrees
        #    here we compute it from an Euler rotation of pi/2 about z
        #    if you have a utility you can replace this.
        pi2 = np.pi / 2.0
        # quaternion [x, y, z, w] for rotation about Z axis by 90°
        target_quat = np.array([0.0, 0.0, np.sin(pi2 / 2.0), np.cos(pi2 / 2.0)])
        obs, reward, done = rotate(env, task, target_quat,
                                   max_steps=200, threshold=0.02, timeout=15.0)
        if done:
            print("[Task] Exited early during rotate.")
            return

        # 2) Move gripper to side position of the drawer
        print(f"[Task] Moving to drawer side position: {drawer_side_pos}")
        obs, reward, done = move(env, task,
                                 target_pos=drawer_side_pos,
                                 approach_distance=0.00,
                                 max_steps=150,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Task] Exited early during side approach.")
            return

        # 3) Move gripper into anchor position on the drawer handle
        print(f"[Task] Approaching drawer anchor position: {drawer_anchor_pos}")
        obs, reward, done = move(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.00,
                                 max_steps=150,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Task] Exited early during anchor approach.")
            return

        # 4) Pick up (grasp) the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.00,
                                 max_steps=100,
                                 threshold=0.005,
                                 approach_axis='z',
                                 timeout=8.0)
        if done:
            print("[Task] Exited early during pick-drawer.")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task,
                                 max_steps=100,
                                 pull_distance=0.15,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Task] Exited early during pull.")
            return

        print("[Task] Drawer should now be open.")
    except Exception as e:
        print(f"[Error] Unexpected exception: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()