# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expect keys for drawer side and anchor positions
        if 'drawer_side_pos' in positions:
            side_pos = positions['drawer_side_pos']
        elif 'drawer_side' in positions:
            side_pos = positions['drawer_side']
        else:
            raise KeyError("Could not find 'drawer_side' in object positions")

        if 'drawer_anchor_pos' in positions:
            anchor_pos = positions['drawer_anchor_pos']
        elif 'drawer_anchor' in positions:
            anchor_pos = positions['drawer_anchor']
        else:
            raise KeyError("Could not find 'drawer_anchor' in object positions")

        # === Oracle Plan ===
        # 1) Rotate gripper to ninety degrees
        target_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Plan] Rotating gripper to 90 degrees")
        obs, reward, done = rotate(
            env, task,
            target_quat=target_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended during rotation")
            return

        # 2) Approach side position of drawer
        print(f"[Plan] Moving to side position: {side_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=side_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended while moving to side")
            return

        # 3) Approach anchor position of drawer
        print(f"[Plan] Moving to anchor position: {anchor_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended while moving to anchor")
            return

        # 4) Grip the drawer handle
        print(f"[Plan] Picking drawer at: {anchor_pos}")
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended during pick-drawer")
            return

        # 5) Pull the drawer open
        print("[Plan] Pulling the drawer")
        obs, reward, done = pull(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.00,
            max_steps=100,
            threshold=0.01,
            approach_axis='x',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended during pull")
            return

        print("===== Plan execution completed successfully =====")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()