import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset and initialize recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve all known positions and angles
        positions = get_object_positions()

        # identify names and key poses
        # assume keys follow the pattern '<drawer_name>_side_pos', '<drawer_name>_anchor_pos'
        # and angles are stored under 'zero_deg', 'ninety_deg'
        drawer_name = 'drawer'
        gripper_name = 'gripper'
        side_pos = positions[f'{drawer_name}_side_pos']
        anchor_pos = positions[f'{drawer_name}_anchor_pos']
        zero_angle = positions['zero_deg']
        ninety_angle = positions['ninety_deg']

        # Step 1: rotate gripper from zero to ninety degrees
        print("[Plan] Rotating gripper from zero to ninety degrees")
        obs, reward, done = rotate(env, task, gripper_name, zero_angle, ninety_angle)
        if done:
            print("[Plan] Task ended during rotate")
            return

        # Step 2: move gripper to the side position of the drawer
        print(f"[Plan] Moving gripper to side position of {drawer_name} at {side_pos}")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] Task ended during move to side")
            return

        # Step 3: move gripper to the anchor position (handle) of the drawer
        print(f"[Plan] Moving gripper to anchor position of {drawer_name} at {anchor_pos}")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Task ended during move to anchor")
            return

        # Step 4: pick the drawer handle
        print(f"[Plan] Picking up the handle of {drawer_name}")
        # using generic pick skill on the handle location
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended during pick")
            return

        # Step 5: pull to open the drawer
        print(f"[Plan] Pulling {drawer_name} to open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Plan] Task ended during pull")
            return

        print("[Plan] Drawer should now be open.")

    except Exception as e:
        print(f"[Error] Exception during plan execution: {e}")
        raise
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()