import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We assume the drawer has two key positions: side approach and anchor (handle)
        drawer_name = 'drawer1'
        side_key = f'{drawer_name}_side_pos'
        anchor_key = f'{drawer_name}_anchor_pos'
        if side_key not in positions or anchor_key not in positions:
            raise KeyError(f"Missing required positions '{side_key}' or '{anchor_key}' in object_positions")

        side_pos = positions[side_key]
        anchor_pos = positions[anchor_key]

        # === Oracle Plan ===

        # 1) Rotate gripper from zero_deg to ninety_deg
        #    quaternion for 90° around Z: [x, y, z, w] = [0, 0, sin(45°), cos(45°)]
        ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        print("[Plan] Rotating gripper to ninety_deg orientation")
        obs, reward, done = rotate(env, task, ninety_deg_quat, max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Plan] Task ended during rotate")
            return

        # 2) Move gripper from current pose to the side position of the drawer
        print(f"[Plan] Moving from current pose to side of {drawer_name} at {side_pos}")
        # We assume move skill signature: move(env, task, from_pos, to_pos, max_steps=..., threshold=..., timeout=...)
        current_pos = task.get_observation().gripper_pose[:3]
        obs, reward, done = move(env, task,
                                 from_pos=current_pos,
                                 to_pos=side_pos,
                                 max_steps=200,
                                 threshold=0.01,
                                 timeout=15.0)
        if done:
            print("[Plan] Task ended during move to side")
            return

        # 3) Move gripper from side position to anchor (handle) position
        print(f"[Plan] Moving from side to anchor of {drawer_name} at {anchor_pos}")
        obs, reward, done = move(env, task,
                                 from_pos=side_pos,
                                 to_pos=anchor_pos,
                                 max_steps=200,
                                 threshold=0.01,
                                 timeout=15.0)
        if done:
            print("[Plan] Task ended during move to anchor")
            return

        # 4) Pick (grasp) the drawer handle
        print(f"[Plan] Picking the handle of {drawer_name}")
        obs, reward, done = pick(env, task,
                                 target_pos=anchor_pos,
                                 approach_distance=0.02,
                                 max_steps=150,
                                 threshold=0.005,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Plan] Task ended during pick")
            return

        # 5) Pull the drawer open
        print(f"[Plan] Pulling {drawer_name} open")
        obs, reward, done = pull(env, task,
                                 target_pos=anchor_pos,
                                 pull_distance=0.1,
                                 max_steps=200,
                                 threshold=0.01,
                                 approach_axis='y',
                                 timeout=10.0)
        if done:
            print("[Plan] Task ended during pull")
            return

        print("[Plan] Drawer should now be open!")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()