import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation to record
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve all object and key positions
        positions = get_object_positions()
        # these keys must be provided by your object_positions module
        # they refer to the side‐approach location and the anchor location of the drawer handle
        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']
        # a quaternion for a ninety degree rotation; adjust if your module provides a different value
        ninety_deg_quat = positions.get('drawer_handle_ninety_quat', [0.7071, 0.0, 0.0, 0.7071])

        # 1) Rotate gripper to 90 degrees so it can approach drawer side
        print("[Task] Rotating gripper to ninety degrees")
        try:
            obs, reward, done = rotate(env, task, ninety_deg_quat)
            if done:
                print("[Task] Done after rotate")
                return
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            return

        # 2) Move to side position of drawer
        print(f"[Task] Moving to drawer side position: {drawer_side_pos}")
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=drawer_side_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Done after move to side")
                return
        except Exception as e:
            print(f"[Error] move to side failed: {e}")
            return

        # 3) Move to anchor position on drawer handle
        print(f"[Task] Moving to drawer anchor position: {drawer_anchor_pos}")
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=drawer_anchor_pos,
                approach_distance=0.05,
                max_steps=100,
                threshold=0.005,
                approach_axis='x',
                timeout=10.0
            )
            if done:
                print("[Task] Done after move to anchor")
                return
        except Exception as e:
            print(f"[Error] move to anchor failed: {e}")
            return

        # 4) Pick (grasp) the drawer handle
        print("[Task] Grasping drawer handle")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=drawer_anchor_pos,
                approach_distance=0.02,
                max_steps=50,
                threshold=0.005,
                approach_axis='x',
                timeout=5.0
            )
            if done:
                print("[Task] Done after pick")
                return
        except Exception as e:
            print(f"[Error] pick failed: {e}")
            return

        # 5) Pull (open) the drawer
        print("[Task] Pulling to open the drawer")
        try:
            obs, reward, done = pull(
                env,
                task,
                target_pos=drawer_anchor_pos,
                distance=0.2,
                max_steps=100,
                threshold=0.01,
                approach_axis='x',
                timeout=10.0
            )
            if done:
                print("[Task] Done after pull")
                return
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return

        print("===== Task completed: drawer opened =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()