import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve positions of all named things in the scene
        positions = get_object_positions()
        print("[Info] Retrieved object positions:", positions.keys())

        # === Exploration Phase: identify missing predicate ===
        print("[Exploration] Inspecting observation attributes for 'is_locked' predicate")
        obs = task.get_observation()
        obs_attrs = [a for a in dir(obs) if not a.startswith('_')]
        print("[Exploration] observation has attributes:", obs_attrs)
        # check if we can directly read lock state
        if hasattr(obs, 'is_locked'):
            drawer_locked = obs.is_locked
            print("[Exploration] Found is_locked flag in observation:", drawer_locked)
        else:
            print("[Exploration] is_locked not in observation; performing small test pull")
            # move to a safe anchor position near drawer handle for testing
            anchor_key = next((k for k in positions if 'anchor' in k), None)
            if anchor_key is None:
                print("[Error] No anchor position found in positions; aborting exploration")
                return
            anchor_pos = positions[anchor_key]
            obs, reward, done = move(
                env, task,
                target_pos=anchor_pos,
                approach_distance=0.10,
                max_steps=50,
                threshold=0.01,
                approach_axis='z',
                timeout=5.0
            )
            if done:
                print("[Exploration] Task ended unexpectedly during move-to-anchor test")
                return
            # attempt a very small pull to see if the drawer budges
            try:
                obs_before = task.get_observation()
                obs, reward, done = pull(
                    env, task,
                    distance=0.01,
                    max_steps=20,
                    threshold=0.001,
                    timeout=2.0
                )
                obs_after = task.get_observation()
                # compare a proxy for drawer position, if available
                if hasattr(obs_before, 'drawer_tip_pose') and hasattr(obs_after, 'drawer_tip_pose'):
                    pos_before = np.array(obs_before.drawer_tip_pose)
                    pos_after = np.array(obs_after.drawer_tip_pose)
                    delta = np.linalg.norm(pos_after - pos_before)
                    drawer_locked = (delta < 1e-3)
                else:
                    # fallback: assume pull exception means locked
                    drawer_locked = False
                print(f"[Exploration] small-pull delta={delta:.4f}, inferred locked={drawer_locked}")
            except Exception as e:
                print("[Exploration] pull test raised exception, inferring locked=True", e)
                drawer_locked = True

        if drawer_locked:
            print("[Task] Drawer appears to be locked; cannot proceed to open.")
            return

        # === Plan Execution Phase ===
        # 1) rotate gripper to the opening orientation (e.g. ninety_deg)
        print("[Plan] Rotating gripper to ninety_deg")
        target_quat = [0., 0., np.sin(np.pi/4), np.cos(np.pi/4)]  # example 90° about z
        obs, reward, done = rotate(env, task, target_quat, max_steps=100, threshold=0.02, timeout=10.0)
        if done:
            print("[Plan] Task ended during rotation")
            return

        # 2) move to side position of drawer
        side_key = next((k for k in positions if 'side' in k), None)
        if side_key is None:
            print("[Error] No side position found; aborting")
            return
        side_pos = positions[side_key]
        print(f"[Plan] Moving to side position: {side_key} -> {side_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=side_pos,
            approach_distance=0.12,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended during move-to-side")
            return

        # 3) move to anchor (drawer handle) position
        print(f"[Plan] Moving to anchor position: {anchor_key} -> {anchor_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.08,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended during move-to-anchor")
            return

        # 4) pick the drawer handle
        print("[Plan] Grasping drawer handle with pick()")
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.05,
            max_steps=80,
            threshold=0.005,
            approach_axis='z',
            timeout=8.0
        )
        if done:
            print("[Plan] Task ended during pick-drawer")
            return

        # 5) pull the drawer open
        print("[Plan] Pulling drawer to open it")
        obs, reward, done = pull(
            env, task,
            distance=0.25,
            max_steps=150,
            threshold=0.01,
            timeout=12.0
        )
        if done:
            print("[Plan] Task ended during pull")
            return

        print("[Task] Drawer should now be open. Plan completed successfully.")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()