import numpy as np
from math import sin, cos, pi

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap for recording
        original_step = task.step
        task.step = recording_step(original_step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve positions of drawer handle, side and anchor
        positions = get_object_positions()
        if 'drawer_handle' not in positions or 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            print("Error: Required position keys not found in get_object_positions()")
            return

        drawer_handle_pos = positions['drawer_handle']
        side_pos = positions['drawer_side_pos']
        anchor_pos = positions['drawer_anchor_pos']

        # Define quaternions for zero and ninety degrees rotation
        zero_deg_quat = [0.0, 0.0, 0.0, 1.0]
        ninety_quat = [0.0, 0.0, sin(pi/4), cos(pi/4)]

        # 1) Ensure gripper is at zero orientation to discover "rotated g zero_deg"
        print("[Task] Calibrating to zero orientation")
        obs, reward, done = rotate(env, task, zero_deg_quat)
        if done:
            print("[Task] Early termination during zero rotation")
            return

        # 2) Rotate to ninety degrees to satisfy move-to-side precondition
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Early termination during ninety rotation")
            return

        # 3) Move to side position of drawer
        print(f"[Task] Moving to drawer side pos: {side_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=side_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after move-to-side")
            return

        # 4) Move to anchor position
        print(f"[Task] Moving to drawer anchor pos: {anchor_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after move-to-anchor")
            return

        # 5) Pick the drawer handle
        print(f"[Task] Picking drawer handle at: {anchor_pos}")
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination during pick")
            return

        # 6) Pull to open the drawer
        print("[Task] Pulling to open drawer")
        obs, reward, done = pull(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.01,
            max_steps=100,
            threshold=0.005,
            approach_axis='x',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination during pull")
            return

        print("[Task] Drawer should now be open!")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()