import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset to initial state
        descriptions, obs = task.reset()
        # Prepare video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve positions of objects
        positions = get_object_positions()
        # Exploration phase: identify any missing predicate in the problem description
        print("===== Exploration Phase: Checking for Missing Predicates =====")
        desc_text = " ".join(descriptions)
        candidate_preds = ['identified', 'temperature-known', 'weight-known', 'durability-known', 'lock-known']
        missing = [p for p in candidate_preds if p not in desc_text]
        if missing:
            print("Missing predicate(s) detected:", missing)
        else:
            print("No missing predicates detected.")
        print("===== End of Exploration Phase =====\n")

        # === Oracle Plan Execution ===
        # 1) Rotate gripper to align for drawer manipulation (e.g., ninety degrees)
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Plan] Rotating gripper to ninety degrees")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Plan] Task ended during rotation")
            return

        # 2) Move robot/gripper to the side position of the drawer
        drawer_side = positions.get('drawer_side_pos')
        if drawer_side is None:
            raise KeyError("drawer_side_pos missing from object_positions")
        print(f"[Plan] Moving to side position of drawer at {drawer_side}")
        obs, reward, done = move(env, task, obs.robot_pose[:3], drawer_side)
        if done:
            print("[Plan] Task ended during move to side")
            return

        # 3) Move robot/gripper to the anchor position of the drawer handle
        drawer_anchor = positions.get('drawer_anchor_pos')
        if drawer_anchor is None:
            raise KeyError("drawer_anchor_pos missing from object_positions")
        print(f"[Plan] Moving to anchor position of drawer at {drawer_anchor}")
        obs, reward, done = move(env, task, drawer_side, drawer_anchor)
        if done:
            print("[Plan] Task ended during move to anchor")
            return

        # 4) Pick the drawer handle
        print("[Plan] Picking drawer handle")
        obs, reward, done = pick(env, task, 'drawer_handle', drawer_anchor)
        if done:
            print("[Plan] Task ended during pick")
            return

        # 5) Pull the drawer open
        print("[Plan] Pulling drawer open")
        obs, reward, done = pull(env, task, 'drawer_handle')
        if done:
            print("[Plan] Task ended during pull")
            return

        print("===== Oracle Plan Completed Successfully =====")

    except Exception as e:
        print(f"[Error] Exception during execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()