import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve all object poses from the scene
        positions = get_object_positions()
        # these keys must match what get_object_positions() returns
        # side and anchor positions of the drawer handle
        side_pose = positions['bottom_side_pos']
        anchor_pose = positions['bottom_anchor_pos']

        # 1) Rotate gripper to align with drawer handle (90 degrees)
        ninety_deg_quat = [0.0, 0.0, 0.7071, 0.7071]
        print("[Task] Rotating gripper to ninety_deg orientation")
        obs, reward, done = rotate(env, task, ninety_deg_quat,
                                   max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Episode ended during rotation")
            return

        # 2) Move gripper to the side position of the drawer
        print(f"[Task] Moving to side position: {side_pose}")
        obs, reward, done = move(env, task,
                                 target_pos=side_pose,
                                 max_steps=200, threshold=0.01, timeout=15.0)
        if done:
            print("[Task] Episode ended during side approach")
            return

        # 3) Move gripper from side to anchor position on the drawer
        print(f"[Task] Moving to anchor position: {anchor_pose}")
        obs, reward, done = move(env, task,
                                 target_pos=anchor_pose,
                                 max_steps=200, threshold=0.01, timeout=15.0)
        if done:
            print("[Task] Episode ended during anchor approach")
            return

        # 4) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task,
                                 max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Episode ended during pull")
            return

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
