import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Validate that the required objects exist
        if 'object_1' not in positions or 'object_2' not in positions:
            print(f"[ERROR] Required objects not found in positions: {list(positions.keys())}")
            return

        object_1_pos = positions['object_1']
        object_2_pos = positions['object_2']

        # === Task Plan ===
        # 1) Pick up object_1
        print("[Task] Picking object_1 at:", object_1_pos)
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=object_1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[ERROR] Exception during pick: {e}")
            return
        if done:
            print("[Task] Task ended after pick!")
            return

        # 2) Place the object at object_2 location
        print("[Task] Placing object_1 at:", object_2_pos)
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=object_2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[ERROR] Exception during place: {e}")
            return
        if done:
            print("[Task] Task ended after place!")
            return

        print("[Task] Plan executed successfully.")
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()