import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # TODO: Replace these keys with the actual names in your environment
        # For example, if your drawer is named "drawer1" and has side/anchor positions recorded:
        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']
        # A quaternion corresponding to a 90° rotation around the required axis.
        # Replace or compute this based on your simulation’s coordinate conventions.
        target_quat_ninety = [0.0, 0.7071, 0.0, 0.7071]  # example: rotation around Y by 90°

        # === Task Plan ===
        # 1) Move gripper to the drawer’s side position
        print("[Task] Moving to drawer side position:", drawer_side_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=drawer_side_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Ended after moving to side position.")
            return

        # 2) Rotate gripper to align with drawer handle
        print("[Task] Rotating gripper to ninety degrees")
        obs, reward, done = rotate(
            env,
            task,
            target_quat_ninety,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Ended during gripper rotation.")
            return

        # 3) Move gripper to the drawer’s anchor position
        print("[Task] Moving to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Ended after moving to anchor position.")
            return

        # 4) Pick (grab) the drawer handle
        print("[Task] Grasping drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.002,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Ended during drawer handle pickup.")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(
            env,
            task,
            max_steps=100,
            threshold=0.02,
            timeout=10.0
        )
        if done:
            print("[Task] Ended during drawer pull.")
            return

        print("[Task] Drawer should now be open.")
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()