import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # get all object positions in the scene
        positions = get_object_positions()
        # assume robot and gripper identifiers are known or default
        robot_name = descriptions.get('robot', 'robot')
        gripper_name = descriptions.get('gripper', 'gripper')

        # === Exploration Phase: Identify objects and their properties ===
        # for each object, move there, pick/drop to trigger identification, weight, durability, temperature effects
        for obj_name, obj_pos in positions.items():
            print(f"[Task] Exploring object '{obj_name}' at position {obj_pos}")
            try:
                # move the robot to the object's location
                obs, reward, done = move(env, task, robot_name, obj_pos)
                if done:
                    print("[Task] Task ended unexpectedly after move.")
                    return

                # attempt to pick the object (will produce weight-known or durability-known)
                obs, reward, done = pick(env, task, obj_name, obj_pos, gripper_name)
                if done:
                    print("[Task] Task ended unexpectedly after pick.")
                    return

                # place it back (so we don't remove it permanently)
                obs, reward, done = place(env, task, obj_name, obj_pos)
                if done:
                    print("[Task] Task ended unexpectedly after place.")
                    return

            except Exception as e:
                print(f"[Task] Warning: could not manipulate '{obj_name}': {e}")

        # === Exploration of rotated predicate ===
        # rotate the gripper to a known orientation to trigger the 'rotated' predicate
        # here we use identity quaternion [0,0,0,1]
        target_quat = [0.0, 0.0, 0.0, 1.0]
        print("[Task] Exploring rotation predicate 'rotated' by rotating to identity quaternion")
        try:
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended unexpectedly during rotate.")
                return
        except Exception as e:
            print(f"[Task] Warning: rotate failed: {e}")

        # === Exploration of pull / lock-known predicate ===
        # attempt to pull on each object to see if it triggers a lock-known or similar effect
        for obj_name in positions.keys():
            print(f"[Task] Exploring pull action on '{obj_name}'")
            try:
                obs, reward, done = pull(env, task, gripper_name, obj_name)
                if done:
                    print("[Task] Task ended unexpectedly during pull.")
                    return
            except Exception as e:
                print(f"[Task] Warning: pull failed on '{obj_name}': {e}")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()