import numpy as np
import math
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve a dictionary of named positions from the scene
        positions = get_object_positions()
        # expected keys in positions: 'drawer_side', 'drawer_anchor'
        if 'drawer_side' not in positions or 'drawer_anchor' not in positions:
            raise KeyError("Expected keys 'drawer_side' and 'drawer_anchor' in positions")

        drawer_side_pos = positions['drawer_side']
        drawer_anchor_pos = positions['drawer_anchor']

        # Step 1: Rotate the gripper to ninety degrees
        # quaternion for a 90° rotation around the appropriate axis
        ninety_deg_quat = [0.0, 0.0, math.sin(math.pi/4), math.cos(math.pi/4)]
        print("[Task] Rotating gripper to ninety_deg_quat:", ninety_deg_quat)
        obs, reward, done = rotate(env, task, ninety_deg_quat, max_steps=100, threshold=0.02, timeout=10.0)
        if done:
            print("[Task] Task ended during rotate")
            return

        # Step 2: Move to the drawer side position
        print("[Task] Moving to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_side_pos,
                                 max_steps=200,
                                 threshold=0.02,
                                 timeout=15.0)
        if done:
            print("[Task] Task ended during move to side")
            return

        # Step 3: Move to the drawer anchor position
        print("[Task] Moving to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_anchor_pos,
                                 max_steps=200,
                                 threshold=0.02,
                                 timeout=15.0)
        if done:
            print("[Task] Task ended during move to anchor")
            return

        # Step 4: Pick the drawer handle
        print("[Task] Picking drawer handle at anchor pos")
        obs, reward, done = pick(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.05,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Task] Task ended during pick")
            return

        # Step 5: Pull to open the drawer
        print("[Task] Pulling to open drawer")
        obs, reward, done = pull(env, task, timeout=5.0)
        if done:
            print("[Task] Task ended during pull")
            return

    except Exception as e:
        print("[Task] Exception encountered:", e)
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()