import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step / get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve positions of all objects in the scene
        positions = get_object_positions()
        print("[Task] Retrieved object positions:", positions.keys())

        # --------------------
        # Exploration Phase
        # --------------------
        # 1) Identify and learn temperature of every object by moving to it
        # 2) Learn weight and durability by picking each object
        # 3) Learn lock state of each drawer by pulling its handle
        #
        # We assume that any key containing 'object' is a generic object,
        # and any key containing 'drawer' is a drawer handle.

        # helper to check for termination
        def check_done(name, action_name, result):
            obs, reward, done = result
            if done:
                print(f"[Task] Terminated early during {action_name} on {name}")
                return True
            return False

        # 1) move to each generic object to trigger identification / temperature-known
        for name, pos in positions.items():
            if 'object' in name:
                print(f"[Exploration] Moving to {name} at {pos}")
                try:
                    result = move(env, task, target_pos=pos, max_steps=100, threshold=0.01, timeout=10.0)
                    if check_done(name, 'move', result):
                        return
                except Exception as e:
                    print(f"[Exploration] move to {name} failed:", e)
                # 2) pick each object to learn weight/durability
                print(f"[Exploration] Picking {name} to learn weight/durability")
                try:
                    result = pick(env, task, target_pos=pos,
                                  approach_distance=0.15, max_steps=100,
                                  threshold=0.01, approach_axis='z', timeout=10.0)
                    if check_done(name, 'pick', result):
                        return
                except Exception as e:
                    print(f"[Exploration] pick of {name} failed:", e)
                # place back so hand is empty again
                print(f"[Exploration] Placing {name} back")
                try:
                    result = place(env, task, target_pos=pos,
                                   max_steps=100, threshold=0.01, timeout=10.0)
                    if check_done(name, 'place', result):
                        return
                except Exception as e:
                    print(f"[Exploration] place of {name} failed:", e)

        # 3) learn lock state of each drawer
        for name, pos in positions.items():
            if 'drawer' in name:
                print(f"[Exploration] Moving to {name} handle at {pos}")
                try:
                    result = move(env, task, target_pos=pos, max_steps=100,
                                  threshold=0.01, timeout=10.0)
                    if check_done(name, 'move', result):
                        return
                except Exception as e:
                    print(f"[Exploration] move to {name} failed:", e)
                print(f"[Exploration] Pulling {name} to learn lock state")
                try:
                    result = pull(env, task, target_pos=pos,
                                  max_steps=100, threshold=0.01, timeout=10.0)
                    if check_done(name, 'pull', result):
                        return
                except Exception as e:
                    print(f"[Exploration] pull of {name} failed:", e)

        print("[Exploration] Completed. Missing predicate identified: lock-known")

        # --------------------
        # (Optional) Main Task Plan
        # You can now proceed to write the plan that uses the newly acquired predicates.
        # For example: open a drawer whose lock state is now known.
        # --------------------

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()