# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap the task stepping for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions from the environment
        positions = get_object_positions()
        print("[Task] Retrieved object positions:", positions)

        # === Exploration Phase: Identify which predicates can be observed ===

        # 1) Identify all objects by moving the robot to their locations
        current_location = None
        identified_objects = []
        for obj_name, obj_pos in positions.items():
            try:
                print(f"[Exploration] Moving to {obj_name} at {obj_pos}")
                obs, reward, done = move(env, task, current_location, obj_pos)
                if done:
                    print("[Exploration] Task ended unexpectedly during move!")
                    return
                current_location = obj_pos
                identified_objects.append(obj_name)
            except Exception as e:
                print(f"[Exploration] move failed for {obj_name}: {e}")

        print("[Exploration] Identified objects:", identified_objects)

        # 2) Learn weight information by picking each identified object
        weight_known = []
        for obj_name in identified_objects:
            try:
                print(f"[Exploration] Picking up {obj_name} to learn weight")
                obs, reward, done = pick(env, task, obj_name, current_location)
                weight_known.append(obj_name)
                print(f"[Exploration] Placing back {obj_name}")
                obs, reward, done = place(env, task, obj_name, current_location)
                if done:
                    print("[Exploration] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Exploration] pick/place failed for {obj_name}: {e}")

        print("[Exploration] Weight-known objects:", weight_known)

        # 3) Learn durability information by picking each object again
        durability_known = []
        for obj_name in identified_objects:
            try:
                print(f"[Exploration] Picking up {obj_name} to learn durability")
                obs, reward, done = pick(env, task, obj_name, current_location)
                durability_known.append(obj_name)
                print(f"[Exploration] Placing back {obj_name}")
                obs, reward, done = place(env, task, obj_name, current_location)
                if done:
                    print("[Exploration] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Exploration] pick/place failed for {obj_name}: {e}")

        print("[Exploration] Durability-known objects:", durability_known)

        # 4) Finally, explore lock status of any drawer handle to discover the missing predicate lock-known
        drawer_handle_key = 'drawer_handle'
        if drawer_handle_key in positions:
            handle_pos = positions[drawer_handle_key]
            try:
                print(f"[Exploration] Moving to drawer handle at {handle_pos}")
                obs, reward, done = move(env, task, current_location, handle_pos)
                if done:
                    print("[Exploration] Task ended during move to drawer handle!")
                    return
                print(f"[Exploration] Grasping drawer handle {drawer_handle_key}")
                obs, reward, done = pick(env, task, drawer_handle_key, handle_pos)
                print(f"[Exploration] Pulling drawer handle to learn lock status")
                obs, reward, done = pull(env, task, drawer_handle_key, handle_pos)
                print("[Exploration] lock-known predicate has been discovered for drawer handle")
            except Exception as e:
                print(f"[Exploration] pull sequence failed: {e}")
        else:
            print("[Exploration] No drawer handle found in positions; cannot learn lock status")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()