import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running a drawer‐opening task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        
        # Initialize video writers if desired
        init_video_writers(obs)
        
        # Wrap task.step and task.get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        
        # Retrieve object positions from the environment
        positions = get_object_positions()
        print("[Task] Retrieved positions:", positions)
        
        # Identify the side, anchor, and handle positions by name matching
        side_pos = None
        anchor_pos = None
        handle_pos = None
        for name, pos in positions.items():
            lname = name.lower()
            if 'side' in lname:
                side_pos = np.array(pos)
            elif 'anchor' in lname or 'pull' in lname:
                anchor_pos = np.array(pos)
            elif 'handle' in lname:
                handle_pos = np.array(pos)
        
        if side_pos is None or anchor_pos is None or handle_pos is None:
            raise ValueError("Could not identify side, anchor or handle positions from object_positions.")
        
        # Step 1: Move to the side of the drawer
        print("[Task] Moving to drawer side position:", side_pos)
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Task ended during move to side.")
            return
        
        # Step 2: Rotate the gripper to ninety degrees
        # Assume starting orientation is 'zero_deg'; goal is 'ninety_deg'
        print("[Task] Rotating gripper from zero_deg to ninety_deg")
        obs, reward, done = rotate(env, task, 'zero_deg', 'ninety_deg')
        if done:
            print("[Task] Task ended during rotate.")
            return
        
        # Step 3: Move to the anchor position at the drawer handle
        print("[Task] Moving to drawer anchor position:", anchor_pos)
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Task ended during move to anchor.")
            return
        
        # Step 4: Grasp (pick) the drawer handle
        print("[Task] Grasping the drawer handle at:", handle_pos)
        # We use a small approach_distance for precision
        obs, reward, done = pick(
            env,
            task,
            target_pos=handle_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during pick of handle.")
            return
        
        # Step 5: Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended during pull.")
            return
        
        print("[Task] Drawer opened successfully!")
    
    except Exception as e:
        print("[Task] Exception encountered:", e)
    finally:
        shutdown_environment(env)
    
    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()