import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running drawer‐opening task.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset environment and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve all known object positions
        positions = get_object_positions()
        # we expect keys like 'drawer_side_pos' and 'drawer_anchor_pos'
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            raise KeyError("Expected 'drawer_side_pos' and 'drawer_anchor_pos' in positions")

        drawer_side_pos = np.array(positions['drawer_side_pos'])
        drawer_anchor_pos = np.array(positions['drawer_anchor_pos'])

        # 1) Rotate the gripper from its current angle to ninety degrees
        print("[Task] Rotating gripper to ninety_deg")
        # assume initial angle is 'zero_deg'; rotate signature is (env, task, from_angle, to_angle)
        obs, reward, done = rotate(env, task, 'zero_deg', 'ninety_deg')
        if done:
            print("[Task] Task ended immediately after rotate")
            return

        # 2) Move the gripper to a side position of the drawer
        print(f"[Task] Moving to drawer side position: {drawer_side_pos}")
        obs, reward, done = move(env, task, drawer_side_pos)
        if done:
            print("[Task] Task ended during move-to-side")
            return

        # 3) Move the gripper from side to anchor position on the drawer
        print(f"[Task] Moving to drawer anchor position: {drawer_anchor_pos}")
        obs, reward, done = move(env, task, drawer_anchor_pos)
        if done:
            print("[Task] Task ended during move-to-anchor")
            return

        # 4) Pull the handle to open the drawer
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended during pull")
            return

        print("[Task] Drawer successfully opened")

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
        raise
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()