import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap to record
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Exploration phase: find any predicate or state we don't yet know ===
        # Here we simply move the robot around to identify visible objects
        print("[Exploration] Identifying objects in all known positions")
        positions = get_object_positions()
        # move to each location in turn to trigger any identification callbacks
        for name, pos in positions.items():
            print(f"[Exploration] Moving to {name} at {pos}")
            obs, reward, done = move(env, task, target_pos=pos, approach_distance=0.2, max_steps=100, threshold=0.01, approach_axis='z', timeout=5.0)
            if done:
                print("[Exploration] Task ended unexpectedly during exploration")
                return

        # now assume we have discovered all relevant objects and their properties

        # === Plan execution to open drawer ===
        # retrieve coordinates for the drawer side position and anchor position from the map
        # keys should match object_positions definitions
        side_key = 'drawer_side_pos'
        anchor_key = 'drawer_anchor_pos'
        if side_key not in positions or anchor_key not in positions:
            raise KeyError(f"Expected keys '{side_key}' and '{anchor_key}' in positions, got {list(positions.keys())}")
        side_pos = positions[side_key]
        anchor_pos = positions[anchor_key]

        # 1) Rotate gripper to ninety_deg orientation
        print("[Task] Rotating gripper to ninety_deg")
        # build a quaternion for a 90° rotation around the z‑axis
        angle = np.pi / 2
        target_quat = [0.0, 0.0, np.sin(angle/2), np.cos(angle/2)]
        obs, reward, done = rotate(env, task, target_quat, max_steps=100, threshold=0.02, timeout=10.0)
        if done:
            print("[Task] Task ended after rotate")
            return

        # 2) Move to the side of the drawer
        print("[Task] Moving to side of drawer at", side_pos)
        obs, reward, done = move(env, task, target_pos=side_pos, approach_distance=0.15, max_steps=150, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended after move-to-side")
            return

        # 3) Move to the anchor position on the drawer handle
        print("[Task] Moving to anchor position at", anchor_pos)
        obs, reward, done = move(env, task, target_pos=anchor_pos, approach_distance=0.05, max_steps=150, threshold=0.005, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended after move-to-anchor")
            return

        # 4) Pick the drawer handle (grasp)
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(env, task, target_pos=anchor_pos, approach_distance=0.02, max_steps=100, threshold=0.005, approach_axis='z', timeout=5.0)
        if done:
            print("[Task] Task ended after pick-drawer")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task, max_steps=80, velocity=0.1, timeout=5.0)
        if done:
            print("[Task] Task ended after pull")
            return

        # Optionally: place or inspect contents of drawer here

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()