import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve positions ===
        positions = get_object_positions()
        # expect keys: 'rubbish' and 'rubbish_table'
        if 'rubbish' not in positions:
            print("Error: 'rubbish' not found in object positions.")
            return
        if 'rubbish_table' not in positions:
            print("Error: 'rubbish_table' not found in object positions.")
            return
        rubbish_pos = positions['rubbish']
        table_pos   = positions['rubbish_table']

        # === Plan Execution ===
        # 1) Pick up the piece of rubbish
        print("[Task] Picking up rubbish at:", rubbish_pos)
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=rubbish_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during pick: {e}")
            return
        if done:
            print("[Task] Task ended immediately after pick.")
            return

        # 2) Place the rubbish onto the table
        print("[Task] Placing rubbish on table at:", table_pos)
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=table_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during place: {e}")
            return
        if done:
            print("[Task] Task ended immediately after place.")
            return

        print("[Task] Successfully picked and placed the rubbish.")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()