# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running a combined-domain task: 
    pick a piece of rubbish from the table, open the drawer, and deposit it inside.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # keys in positions are assumed to include:
        #   'rubbish_table'       : 3D position of the piece of rubbish on the table
        #   'drawer_side_pos'     : a side approach position for the drawer handle
        #   'drawer_anchor_pos'   : a front‐facing anchor position for the handle
        #   'drawer_inside_pos'   : a drop-in position inside the opened drawer
        rubbish_pos        = positions['rubbish_table']
        drawer_side_pos    = positions['drawer_side_pos']
        drawer_anchor_pos  = positions['drawer_anchor_pos']
        drawer_inside_pos  = positions['drawer_inside_pos']

        # === 1) Pick up the rubbish from the table ===
        print("[Task] Picking up rubbish at:", rubbish_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended early during pick.")
            return

        # === 2) Rotate gripper so it can grasp drawer handle ===
        print("[Task] Rotating gripper to ninety degrees.")
        # we assume the gripper starts at some 'zero_deg' orientation
        obs, reward, done = rotate(env, task, 'zero_deg', 'ninety_deg')
        if done:
            print("[Task] Task ended early during rotate.")
            return

        # === 3) Move gripper to the side of the drawer ===
        print("[Task] Moving to side position of drawer.")
        obs, reward, done = move(env, task, drawer_side_pos)
        if done:
            print("[Task] Task ended early during move-to-side.")
            return

        # === 4) Move gripper to anchor position on the drawer handle ===
        print("[Task] Moving to anchor position of drawer handle.")
        obs, reward, done = move(env, task, drawer_anchor_pos)
        if done:
            print("[Task] Task ended early during move-to-anchor.")
            return

        # === 5) Pull the drawer open ===
        print("[Task] Pulling the drawer open.")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended early during pull.")
            return

        # === 6) Place the rubbish inside the opened drawer ===
        print("[Task] Placing rubbish inside drawer at:", drawer_inside_pos)
        obs, reward, done = place(env, task, drawer_inside_pos)
        if done:
            print("[Task] Task ended early during place.")
            return

        print("[Task] Successfully completed the pick‐and‐deposit sequence.")

    except Exception as e:
        # Catch any unexpected errors and log them
        print(f"[Error] Exception during task execution: {e}")
        raise

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()