import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        for name, pos in positions.items():
            print(f"[Info] {name} @ {pos}")

        # === Exploration Phase: identify missing predicate ===
        print("[Phase] Exploration: identifying lock-known predicate")
        lock_predicate_identified = False
        for name, pos in positions.items():
            print(f"[Explore] moving to {name}")
            try:
                obs, reward, done = move(env, task, target_pos=pos)
            except Exception as e:
                print(f"[Explore] move failed for {name}: {e}")
                continue
            if done:
                print("[Explore] Task ended during move.")
                return

            print(f"[Explore] picking {name} to learn properties")
            try:
                obs, reward, done = pick(env, task, target_pos=pos)
            except Exception as e:
                print(f"[Explore] pick failed for {name}: {e}")
            if done:
                print("[Explore] Task ended during pick.")
                return

            # assume drawer handle name contains 'drawer'
            if 'drawer' in name and not lock_predicate_identified:
                print(f"[Explore] pulling on {name} to learn lock state")
                try:
                    obs, reward, done = pull(env, task)
                except Exception as e:
                    print(f"[Explore] pull failed: {e}")
                if done:
                    print("[Explore] Task ended during pull.")
                    return
                print("[Explore] lock-known predicate achieved")
                lock_predicate_identified = True
                break

        if not lock_predicate_identified:
            print("[Warning] Could not identify lock state during exploration, proceeding anyway")

        # === Oracle Plan Execution: open the drawer ===
        print("[Phase] Oracle plan execution: open drawer")
        # 1) Rotate gripper from its current angle to ninety_deg
        try:
            obs, reward, done = rotate(env, task, 'zero_deg', 'ninety_deg')
        except Exception as e:
            print(f"[Plan] rotate failed: {e}")
            return
        if done:
            print("[Plan] Task ended after rotate.")
            return

        # 2) Move gripper to side position of drawer
        side_key = next((k for k in positions if 'side' in k), None)
        if side_key is None:
            print("[Plan] side position for drawer not found.")
            return
        side_pos = positions[side_key]
        try:
            obs, reward, done = move(env, task, target_pos=side_pos)
        except Exception as e:
            print(f"[Plan] move to side failed: {e}")
            return
        if done:
            print("[Plan] Task ended after move to side.")
            return

        # 3) Move gripper to anchor position (assume named accordingly)
        anchor_key = next((k for k in positions if 'anchor' in k), None)
        if anchor_key is None:
            print("[Plan] anchor position for drawer not found.")
            return
        anchor_pos = positions[anchor_key]
        try:
            obs, reward, done = move(env, task, target_pos=anchor_pos)
        except Exception as e:
            print(f"[Plan] move to anchor failed: {e}")
            return
        if done:
            print("[Plan] Task ended after move to anchor.")
            return

        # 4) Pick drawer handle at anchor position
        try:
            obs, reward, done = pick(env, task, target_pos=anchor_pos)
        except Exception as e:
            print(f"[Plan] pick drawer handle failed: {e}")
            return
        if done:
            print("[Plan] Task ended after pick-drawer.")
            return

        # 5) Pull drawer open
        try:
            obs, reward, done = pull(env, task)
        except Exception as e:
            print(f"[Plan] pull drawer open failed: {e}")
            return
        if done:
            print("[Plan] Task ended after pull drawer.")
            return

        print("[Result] Drawer opened successfully.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()