import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # The problem declares item1, item2, item3 as objects
        try:
            item1_pos = positions['item1']
            item2_pos = positions['item2']
            item3_pos = positions['item3']
        except KeyError as e:
            print(f"[Error] Missing position for {e.args[0]}")
            return

        # If there is a drawer, we expect its side and anchor positions
        try:
            drawer_side_pos = positions['drawer_side_pos']
            drawer_anchor_pos = positions['drawer_anchor_pos']
        except KeyError:
            print("[Warning] Drawer positions not found; skipping drawer opening steps")
            drawer_side_pos = None
            drawer_anchor_pos = None

        # === Exploration Phase: Identify properties of each item ===
        for name, pos in [('item1', item1_pos), ('item2', item2_pos), ('item3', item3_pos)]:
            print(f"[Task] Moving to {name} at {pos}")
            obs, reward, done = move(env, task, pos)
            if done:
                print(f"[Task] Task ended while moving to {name}")
                return
            print(f"[Task] Picking up {name}")
            obs, reward, done = pick(env, task, name, pos)
            if done:
                print(f"[Task] Task ended while picking {name}")
                return
            print(f"[Task] Placing {name} back")
            obs, reward, done = place(env, task, name, pos)
            if done:
                print(f"[Task] Task ended while placing {name}")
                return

        # === Drawer Opening Phase ===
        if drawer_side_pos and drawer_anchor_pos:
            # Rotate gripper to ninety degrees (example quaternion)
            ninety_deg_quat = [0.0, 0.7071, 0.0, 0.7071]
            print("[Task] Rotating gripper to ninety degrees")
            obs, reward, done = rotate(env, task, ninety_deg_quat)
            if done:
                print("[Task] Task ended during rotation")
                return

            print("[Task] Moving gripper to side of drawer")
            obs, reward, done = move(env, task, drawer_side_pos)
            if done:
                print("[Task] Task ended while moving to drawer side")
                return

            print("[Task] Moving gripper to drawer anchor position")
            obs, reward, done = move(env, task, drawer_anchor_pos)
            if done:
                print("[Task] Task ended while moving to drawer anchor")
                return

            # Grasp the drawer handle (modeled as a pick-drawer action)
            print("[Task] Grasping drawer handle")
            obs, reward, done = pick(env, task, 'drawer_handle', drawer_anchor_pos)
            if done:
                print("[Task] Task ended while grasping drawer handle")
                return

            print("[Task] Pulling drawer open")
            obs, reward, done = pull(env, task)
            if done:
                print("[Task] Task ended while pulling drawer")
                return

        print("===== Task Plan Completed Successfully =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()