import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Skeleton task to open a drawer by rotating, moving, grasping, and pulling.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get predefined positions and orientations
        positions = get_object_positions()
        # expected keys in positions dict:
        #   'drawer_side_pos'       -> xyz tuple for the side approach
        #   'drawer_anchor_pos'     -> xyz tuple for the anchor handle
        #   'drawer_ninety_quat'    -> quaternion (xyzw) for 90° gripper rotation
        try:
            side_pos = positions['drawer_side_pos']
            anchor_pos = positions['drawer_anchor_pos']
            ninety_quat = positions['drawer_ninety_quat']
        except KeyError as e:
            print(f"[Error] Missing expected object position key: {e}")
            return

        # 1) Rotate gripper to 90 degrees
        print("[Task] Rotating gripper to ninety_deg orientation")
        try:
            obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_quat,
                max_steps=200,
                threshold=0.02,
                timeout=15.0
            )
            if done:
                print("[Task] Episode ended during rotate.")
                return
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            return

        # 2) Move to side position of the drawer
        print("[Task] Moving to drawer side position at", side_pos)
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=side_pos,
                approach_distance=0.05,
                max_steps=150,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Episode ended during move-to-side.")
                return
        except Exception as e:
            print(f"[Error] move to side failed: {e}")
            return

        # 3) Move to anchor position (drawer handle)
        print("[Task] Moving to drawer anchor position at", anchor_pos)
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=anchor_pos,
                approach_distance=0.02,
                max_steps=150,
                threshold=0.005,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Episode ended during move-to-anchor.")
                return
        except Exception as e:
            print(f"[Error] move to anchor failed: {e}")
            return

        # 4) Pick (grasp) the drawer handle
        print("[Task] Grasping drawer handle")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=anchor_pos,
                approach_distance=0.0,
                max_steps=100,
                threshold=0.005,
                approach_axis='z',
                timeout=8.0
            )
            if done:
                print("[Task] Episode ended during pick.")
                return
        except Exception as e:
            print(f"[Error] pick failed: {e}")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer")
        try:
            obs, reward, done = pull(
                env,
                task,
                # if pull requires a target, reuse anchor_pos
                target_pos=anchor_pos,
                max_steps=100,
                timeout=5.0
            )
            if done:
                print("[Task] Episode ended during pull.")
                return
        except TypeError:
            # if pull does not need kwargs
            try:
                obs, reward, done = pull(env, task)
                if done:
                    print("[Task] Episode ended during pull.")
                    return
            except Exception as e2:
                print(f"[Error] pull failed: {e2}")
                return
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return

        print("===== Task Completed: Drawer is open =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()