import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running an exploratory+execution task.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step & get_observation
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # get the dictionary of object positions and named handles
        positions = get_object_positions()
        print("[Info] Retrieved positions:", positions.keys())

        # ---- Safety check: ensure we have at least one object to explore ----
        if not positions:
            print("[Error] No objects detected in workspace; aborting.")
            return

        # ---- Exploration phase: visit each location, identify & characterize objects ----
        # we assume that move(...) will mark `identified` and `temperature-known` in the observation
        current_loc = None
        for name, pos in positions.items():
            print(f"[Explore] Moving to {name} at {pos}")
            try:
                obs, reward, done = move(env, task, current_loc, name)
            except Exception as e:
                print(f"[Warning] move to {name} failed: {e}")
                continue
            if done:
                print("[Explore] Task ended prematurely after move")
                return
            current_loc = name

            print(f"[Explore] Picking {name} to learn weight/durability")
            try:
                obs, reward, done = pick(env, task, name, current_loc)
            except Exception as e:
                print(f"[Warning] pick of {name} failed: {e}")
            if done:
                print("[Explore] Task ended prematurely after pick")
                return

            # if this object is a drawer-handle, attempt to pull it to learn its lock state
            if "drawer" in name or "handle" in name:
                print(f"[Explore] Attempting pull on {name} to learn lock status")
                try:
                    obs, reward, done = pull(env, task, name, current_loc)
                except Exception as e:
                    print(f"[Info] pull not applicable on {name}: {e}")
                if done:
                    print("[Explore] Task ended prematurely after pull")
                    return

        # ---- Execution phase: open drawer, pick target object from inside, place to drop zone ----
        # assume positions contains entries 'drawer_handle', 'target_object', 'drop_zone'
        for key in ("drawer_handle", "target_object", "drop_zone"):
            if key not in positions:
                print(f"[Error] Missing required position entry: {key}; aborting.")
                return

        handle_loc = positions["drawer_handle"]
        obj_name = "target_object"
        obj_loc = positions[obj_name]
        drop_loc = positions["drop_zone"]

        # orient gripper for drawer opening
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Goal] Rotating gripper to ninety_deg for drawer approach")
        try:
            obs, reward, done = rotate(env, task, ninety_deg_quat, max_steps=200, threshold=0.02, timeout=8.0)
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            return
        if done:
            print("[Goal] Task ended during initial rotate")
            return

        # approach handle
        print("[Goal] Moving gripper to drawer handle")
        try:
            obs, reward, done = move(env, task, current_loc, "drawer_side")
        except Exception as e:
            print(f"[Error] move to drawer side failed: {e}")
            return
        if done:
            print("[Goal] Task ended during move-to-side")
            return

        print("[Goal] Moving gripper to drawer anchor")
        try:
            obs, reward, done = move(env, task, "drawer_side", "drawer_anchor")
        except Exception as e:
            print(f"[Error] move to drawer anchor failed: {e}")
            return
        if done:
            print("[Goal] Task ended during move-to-anchor")
            return

        # grasp handle
        print("[Goal] Picking drawer handle")
        try:
            obs, reward, done = pick(env, task, "drawer_handle", "drawer_anchor")
        except Exception as e:
            print(f"[Error] pick-drawer failed: {e}")
            return
        if done:
            print("[Goal] Task ended during pick-drawer")
            return

        # pull drawer open
        print("[Goal] Pulling drawer open")
        try:
            obs, reward, done = pull(env, task, "drawer_handle", "drawer_anchor")
        except Exception as e:
            print(f"[Error] pull drawer failed: {e}")
            return
        if done:
            print("[Goal] Task ended during pull")
            return

        # retrieve the object from inside
        print(f"[Goal] Picking up target object '{obj_name}' at {obj_loc}")
        try:
            obs, reward, done = pick(env, task, obj_name, obj_loc)
        except Exception as e:
            print(f"[Error] pick of target object failed: {e}")
            return
        if done:
            print("[Goal] Task ended during object pick")
            return

        # place into drop zone
        print(f"[Goal] Placing '{obj_name}' into drop zone at {drop_loc}")
        try:
            obs, reward, done = place(env, task, obj_name, drop_loc)
        except Exception as e:
            print(f"[Error] place failed: {e}")
            return
        if done:
            print("[Goal] Task ended during place")
            return

        print("[Success] Completed oracle plan successfully.")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
