import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We expect keys like 'drawer_side_pos', 'drawer_anchor_pos'
        side_keys = [k for k in positions if 'side' in k]
        anchor_keys = [k for k in positions if 'anchor' in k]
        if not side_keys or not anchor_keys:
            print("Error: Could not find side or anchor positions in object_positions.")
            return
        side_pos = positions[side_keys[0]]
        anchor_pos = positions[anchor_keys[0]]
        print(f"[Info] Using side position '{side_keys[0]}' -> {side_pos}")
        print(f"[Info] Using anchor position '{anchor_keys[0]}' -> {anchor_pos}")

        # === Oracle Plan Execution ===

        # 1) Rotate gripper to ninety degrees orientation for side approach
        # Here we choose a quaternion representing 90° around z-axis
        target_quat = np.array([0.0, 0.0, 0.7071, 0.7071])
        print("[Plan] Rotating gripper to 90° for side approach")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Task ended prematurely during rotate.")
            return

        # 2) Move to side position of drawer
        print(f"[Plan] Moving to side position at {side_pos}")
        obs, reward, done = move(env,
                                 task,
                                 target_pos=side_pos,
                                 max_steps=100,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Plan] Task ended prematurely during move-to-side.")
            return

        # 3) Move from side to anchor position (handle) of drawer
        print(f"[Plan] Moving to anchor position at {anchor_pos}")
        obs, reward, done = move(env,
                                 task,
                                 target_pos=anchor_pos,
                                 max_steps=100,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Plan] Task ended prematurely during move-to-anchor.")
            return

        # 4) Pick the drawer handle at the anchor position
        print(f"[Plan] Picking drawer handle at {anchor_pos}")
        obs, reward, done = pick(env,
                                 task,
                                 target_pos=anchor_pos,
                                 approach_distance=0.02,
                                 max_steps=100,
                                 threshold=0.005,
                                 timeout=10.0)
        if done:
            print("[Plan] Task ended prematurely during pick.")
            return

        # 5) Pull to open the drawer
        print("[Plan] Pulling drawer open")
        obs, reward, done = pull(env,
                                 task,
                                 target_pos=anchor_pos,
                                 max_steps=100,
                                 threshold=0.005,
                                 timeout=10.0)
        if done:
            print("[Plan] Task ended prematurely during pull.")
            return

        print("[Plan] Drawer should now be open. Plan succeeded.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()