import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset and wrap for recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve object positions from the scene
        positions = get_object_positions()
        # We expect at least these keys in positions:
        #   'drawer_handle', 'drawer_anchor', 'drawer_side'
        if 'drawer_handle' not in positions or \
           'drawer_anchor' not in positions or \
           'drawer_side' not in positions:
            raise KeyError("Expected keys 'drawer_handle', 'drawer_anchor', 'drawer_side' in positions.")

        drawer_handle_pos = positions['drawer_handle']
        drawer_side_pos   = positions['drawer_side']
        drawer_anchor_pos = positions['drawer_anchor']

        # STEP 1: Move close to the drawer handle
        print("[Task] Moving to drawer handle at:", drawer_handle_pos)
        obs, reward, done = move(
            env, task,
            target_pos=drawer_handle_pos,
            approach_distance=0.10,
            max_steps=200,
            threshold=0.01,
            approach_axis='z',
            timeout=15.0
        )
        if done:
            print("[Task] Episode ended early after move to handle.")
            return

        # STEP 2: Pick the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(
            env, task,
            target_pos=drawer_handle_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended early after picking handle.")
            return

        # STEP 3: Rotate gripper to ninety degrees so it can pull
        # Construct a quaternion representing a 90° rotation around the Y axis
        sin45 = np.sin(np.pi/4)
        cos45 = np.cos(np.pi/4)
        ninety_deg_quat = [0.0, sin45, 0.0, cos45]
        print("[Task] Rotating gripper 90 degrees for pulling action")
        obs, reward, done = rotate(
            env, task,
            target_quat=ninety_deg_quat,
            max_steps=100,
            threshold=0.03,
            timeout=8.0
        )
        if done:
            print("[Task] Episode ended early during rotate.")
            return

        # STEP 4: Move to the side of the drawer (so the gripper is aligned with the anchor)
        print("[Task] Moving to side position of drawer:", drawer_side_pos)
        obs, reward, done = move(
            env, task,
            target_pos=drawer_side_pos,
            approach_distance=0.05,
            max_steps=200,
            threshold=0.01,
            approach_axis='xy',
            timeout=12.0
        )
        if done:
            print("[Task] Episode ended early after move to side.")
            return

        # STEP 5: Move in to the anchor point to prepare for pull
        print("[Task] Moving to anchor point of drawer:", drawer_anchor_pos)
        obs, reward, done = move(
            env, task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.02,
            max_steps=150,
            threshold=0.005,
            approach_axis='x',
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended early after move to anchor.")
            return

        # STEP 6: Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(
            env, task,
            target_pos=drawer_anchor_pos,
            max_steps=80,
            threshold=0.01,
            timeout=6.0
        )
        if done:
            print("[Task] Episode ended after pulling drawer.")
            return

        # Task completed
        print("===== Drawer opening sequence completed successfully =====")

    except KeyError as e:
        print(f"[Error] Missing expected object position: {e}")
    except Exception as e:
        print(f"[Error] Unexpected exception during task: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()