import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset environment and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve object positions from the scene
        positions = get_object_positions()
        # assume keys in positions include:
        #   'drawer_side_pos', 'drawer_anchor_pos'
        side_pos = positions.get('drawer_side_pos')
        anchor_pos = positions.get('drawer_anchor_pos')
        if side_pos is None or anchor_pos is None:
            raise KeyError("Missing required drawer positions in object_positions")

        # 1) Rotate gripper to side orientation (90° about Z)
        ninety_deg_quat = [0., 0., np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Task] Rotating gripper to 90° around Z axis")
        obs, reward, done = rotate(
            env, task, ninety_deg_quat,
            max_steps=200, threshold=0.1, timeout=30.0
        )
        if done:
            print("[Task] Done early during rotate")
            return

        # 2) Move gripper to a side approach position
        print(f"[Task] Moving to side position {side_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=side_pos,
            approach_distance=0.05,
            max_steps=200,
            threshold=0.02,
            approach_axis='z',
            timeout=30.0
        )
        if done:
            print("[Task] Done early during move-to-side")
            return

        # 3) Move gripper from side to anchor on drawer handle
        print(f"[Task] Moving to anchor position {anchor_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=200,
            threshold=0.01,
            approach_axis='xyz',
            timeout=30.0
        )
        if done:
            print("[Task] Done early during move-to-anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Picking drawer handle")
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.005,
            max_steps=200,
            threshold=0.005,
            approach_axis='x',
            timeout=30.0
        )
        if done:
            print("[Task] Done early during pick-drawer")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling open the drawer")
        # pull signature may vary; here we assume simple call
        obs, reward, done = pull(env, task, max_steps=100, timeout=30.0)
        if done:
            print("[Task] Done early during pull")
            return

        print("[Task] Drawer opening sequence complete")
    except Exception as e:
        print(f"[Task] Exception during execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()