import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Actual objects in this task:
        # 'waypoint1' is a safe approach position for the drawer handle
        # 'item1' is the trash we need to throw away
        # 'item2' and 'item3' should remain untouched
        # 'bin' is the disposal container
        waypoint1_pos = np.array(positions['waypoint1'])
        trash_pos     = np.array(positions['item1'])
        bin_pos       = np.array(positions['bin'])

        # === Task Plan ===

        # 1) Approach drawer handle at waypoint1 and open drawer via pull
        print("[Task] Approaching drawer handle at:", waypoint1_pos)
        obs, reward, done = move(env, task, target_pos=waypoint1_pos)
        if done:
            print("[Task] Task ended prematurely during drawer approach!")
            return

        print("[Task] Pulling to open the drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended prematurely during drawer pull!")
            return

        # 2) Approach the trash item inside the open drawer
        print("[Task] Approaching trash item at:", trash_pos)
        obs, reward, done = move(env, task, target_pos=trash_pos)
        if done:
            print("[Task] Task ended prematurely during trash approach!")
            return

        # 3) Pick up the trash
        print("[Task] Picking up trash")
        obs, reward, done = pick(
            env,
            task,
            target_pos=trash_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended prematurely during pick!")
            return

        # 4) Transport trash to bin
        print("[Task] Moving to bin at:", bin_pos)
        obs, reward, done = move(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Task ended prematurely during move to bin!")
            return

        # 5) Place trash in bin
        print("[Task] Placing trash in bin")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended prematurely during place!")
            return

        print("===== Task Completed Successfully =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

if __name__ == "__main__":
    run_skeleton_task()