from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
from skill_code import pick, place, move, rotate, pull
import numpy as np

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve key positions from the scene
        positions = get_object_positions()
        side_key = next((k for k in positions if 'side' in k), None)
        anchor_key = next((k for k in positions if 'anchor' in k), None)
        if side_key is None or anchor_key is None:
            raise KeyError("Could not find both side and anchor positions in object_positions()")
        side_pos = positions[side_key]
        anchor_pos = positions[anchor_key]

        # 1) Orient the gripper to ninety_deg
        #    ninety_deg quaternion around Z: [0,0,sin(pi/4),cos(pi/4)]
        ninety_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Task] Rotating gripper to ninety_deg orientation")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Episode ended during rotate")
            return

        # 2) Move the gripper to the side position of the drawer
        print("[Task] Moving to side position:", side_pos)
        try:
            obs, reward, done = move(env, task, side_pos)
        except Exception as e:
            print(f"[Error] move to side failed: {e}")
            return
        if done:
            print("[Task] Episode ended after move-to-side")
            return

        # 3) Move the gripper to the anchor position (handle) of the drawer
        print("[Task] Moving to anchor position:", anchor_pos)
        try:
            obs, reward, done = move(env, task, anchor_pos)
        except Exception as e:
            print(f"[Error] move to anchor failed: {e}")
            return
        if done:
            print("[Task] Episode ended after move-to-anchor")
            return

        # 4) Grasp the drawer handle
        print("[Task] Grasping drawer handle at anchor")
        try:
            obs, reward, done = pick(env, task, anchor_pos)
        except Exception as e:
            print(f"[Error] pick at anchor failed: {e}")
            return
        if done:
            print("[Task] Episode ended after pick")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer open")
        try:
            obs, reward, done = pull(env, task)
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return
        if done:
            print("[Task] Episode ended after pull")
            return

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()