import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step/get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # get the positions of all relevant objects and waypoints
        positions = get_object_positions()

        # Replace these keys with the actual ones in your environment
        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']
        inside_object_key = 'object_1'          # e.g. the object inside the drawer
        drop_location_key = 'drop_location'     # where to place the object

        if inside_object_key not in positions or drop_location_key not in positions:
            print(f"Error: Missing keys in positions: {inside_object_key} or {drop_location_key}")
            return

        inside_object_pos = positions[inside_object_key]
        drop_pos = positions[drop_location_key]

        # 1) Rotate gripper so it can approach the drawer handle
        #    target_quat must correspond to the ninety_deg orientation
        target_quat = [0.0, 0.0, 0.7071, 0.7071]  # placeholder; replace with actual quaternion if known
        print("[Task] Rotating gripper to drawer orientation")
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Task] Early termination after rotate")
            return

        # 2) Move to the side position of the drawer handle
        print("[Task] Moving to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task, drawer_side_pos)
        if done:
            print("[Task] Early termination after move to side")
            return

        # 3) Move to the anchor position on the drawer handle
        print("[Task] Moving to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task, drawer_anchor_pos)
        if done:
            print("[Task] Early termination after move to anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("[Task] Early termination after pick-drawer")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("[Task] Early termination after pull")
            return

        # 6) Move to the object inside the drawer
        print(f"[Task] Moving to inside object '{inside_object_key}':", inside_object_pos)
        obs, reward, done = move(env, task, inside_object_pos)
        if done:
            print("[Task] Early termination after move to object")
            return

        # 7) Pick the object inside the drawer
        print(f"[Task] Picking object '{inside_object_key}'")
        obs, reward, done = pick(env, task, target_pos=inside_object_pos)
        if done:
            print("[Task] Early termination after pick object")
            return

        # 8) Move to the drop location
        print("[Task] Moving to drop location:", drop_pos)
        obs, reward, done = move(env, task, drop_pos)
        if done:
            print("[Task] Early termination after move to drop")
            return

        # 9) Place the object at the drop location
        print("[Task] Placing object at drop location")
        obs, reward, done = place(env, task, target_pos=drop_pos)
        if done:
            print("[Task] Early termination after place")
            return

        print("===== Task completed successfully =====")

    finally:
        shutdown_environment(env)
        print("===== Shutdown environment =====")

if __name__ == "__main__":
    run_skeleton_task()