import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve object positions
        positions = get_object_positions()
        # Identify the side and anchor positions for the drawer from the dictionary
        try:
            drawer_side_pos = positions['drawer_side_pos']
            drawer_anchor_pos = positions['drawer_anchor_pos']
        except KeyError:
            # fallback: search by substring
            drawer_side_pos = next(v for k, v in positions.items() if 'side' in k)
            drawer_anchor_pos = next(v for k, v in positions.items() if 'anchor' in k)

        # Define target orientation quaternion for a 90° rotation about z
        ninety_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]

        # 1) Rotate the gripper to the side orientation
        print("[Task] Rotating gripper to ninety degrees orientation")
        obs, reward, done = rotate(env, task, ninety_quat, max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Terminated after rotate")
            return

        # 2) Move to the side position of the drawer
        print(f"[Task] Moving to side position: {drawer_side_pos}")
        obs, reward, done = move(env, task, target_pos=drawer_side_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Terminated after move to side")
            return

        # 3) Move to the anchor position on the drawer handle
        print(f"[Task] Moving to anchor position: {drawer_anchor_pos}")
        obs, reward, done = move(env, task, target_pos=drawer_anchor_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Terminated after move to anchor")
            return

        # 4) Pick (grasp) the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.02,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Task] Terminated after pick")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Terminated after pull")
            return

        print("[Task] Successfully opened the drawer!")
    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()