import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset environment and get initial observation
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get positions of objects in the scene
        positions = get_object_positions()
        # assume there are two objects named 'object_1' and 'object_2'
        object_1_pos = positions.get('object_1')
        object_2_pos = positions.get('object_2')
        # if there's a drawer handle or anchor position in the scene
        drawer_handle_pos = positions.get('drawer_handle')
        side_pos = positions.get('drawer_side_pos')
        anchor_pos = positions.get('drawer_anchor_pos')

        # Define the oracle plan as a sequence of (skill_name, params) entries
        plan = [
            # Explore and manipulate object_1
            ('move',    {'target_pos': object_1_pos}),
            ('pick',    {'target_pos': object_1_pos,
                         'approach_distance': 0.15,
                         'max_steps': 100,
                         'threshold': 0.01,
                         'approach_axis': 'z',
                         'timeout': 10.0}),
            ('place',   {'target_pos': object_1_pos,
                         'place_distance': 0.10,
                         'max_steps': 100,
                         'threshold': 0.01,
                         'place_axis': 'z',
                         'timeout': 10.0}),
            # Explore and manipulate object_2
            ('move',    {'target_pos': object_2_pos}),
            ('pick',    {'target_pos': object_2_pos,
                         'approach_distance': 0.15,
                         'max_steps': 100,
                         'threshold': 0.01,
                         'approach_axis': 'z',
                         'timeout': 10.0}),
            ('place',   {'target_pos': object_2_pos,
                         'place_distance': 0.10,
                         'max_steps': 100,
                         'threshold': 0.01,
                         'place_axis': 'z',
                         'timeout': 10.0}),
            # Open the drawer: rotate to side position, move to anchor, pick handle, pull
            ('rotate',  {'target_quat': [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]}),
            ('move',    {'target_pos': side_pos}),
            ('move',    {'target_pos': anchor_pos}),
            ('pick',    {'target_pos': anchor_pos,
                         'approach_distance': 0.05,
                         'max_steps': 100,
                         'threshold': 0.005,
                         'approach_axis': 'x',
                         'timeout': 5.0}),
            ('pull',    {})
        ]

        # execute each step in the plan
        for step_index, (skill_name, params) in enumerate(plan):
            print(f"[Plan] Step {step_index+1}: {skill_name} with params {params}")
            try:
                if skill_name == 'move':
                    obs, reward, done = move(env, task, **params)
                elif skill_name == 'pick':
                    obs, reward, done = pick(env, task, **params)
                elif skill_name == 'place':
                    obs, reward, done = place(env, task, **params)
                elif skill_name == 'rotate':
                    target_quat = params.get('target_quat')
                    obs, reward, done = rotate(env, task, target_quat)
                elif skill_name == 'pull':
                    obs, reward, done = pull(env, task)
                else:
                    raise ValueError(f"Unknown skill: {skill_name}")

                if done:
                    print(f"[Task] Terminated early after step {step_index+1}.")
                    return

            except Exception as e:
                print(f"[Error] Exception during '{skill_name}' at step {step_index+1}: {e}")
                # decide whether to continue or abort; here we abort
                return

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()