import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve known object positions
        positions = get_object_positions()
        # names in positions dict must match keys returned by object_positions
        # expecting keys 'drawer_side_pos' and 'drawer_anchor_pos'
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            print("Error: Required drawer positions not found in object_positions.")
            return
        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']

        # define a 90° rotation quaternion (around gripper axis)
        # this may need adjustment depending on RLBench conventions
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]

        # 1) rotate end-effector to face drawer handle
        print("[Task] Rotating gripper to ninety degrees")
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("Task ended during rotate.")
            return

        # 2) move to the drawer side position
        print("[Task] Moving to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task, target_pos=drawer_side_pos)
        if done:
            print("Task ended during move-to-side.")
            return

        # 3) move to the anchor/handle position
        print("[Task] Moving to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("Task ended during move-to-anchor.")
            return

        # 4) pick the drawer handle
        print("[Task] Picking drawer handle")
        obs, reward, done = pick(env, task, target_pos=drawer_anchor_pos,
                                 approach_distance=0.02, max_steps=100,
                                 threshold=0.01, approach_axis='z', timeout=5.0)
        if done:
            print("Task ended during pick-drawer.")
            return

        # 5) pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("Task ended during pull.")
            return

        print("Task completed: drawer should now be open.")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()