import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We expect keys for the drawer task:
        #   'drawer_side'   : a pose just at the side of the drawer handle
        #   'drawer_anchor' : a pose at the anchor point for grasping the handle
        if 'drawer_side' not in positions or 'drawer_anchor' not in positions:
            raise KeyError("Expected 'drawer_side' and 'drawer_anchor' in positions")

        side_pos = positions['drawer_side']
        anchor_pos = positions['drawer_anchor']

        # === Oracle Plan Execution ===

        # 1) Rotate gripper from zero_deg to ninety_deg
        #    quaternion for a 90-degree rotation about the Z axis: [x, y, z, w]
        target_quat = [0.0, 0.0, 0.7071, 0.7071]
        print("[Task] Rotating gripper to 90 degrees")
        obs, reward, done = rotate(env, task, target_quat=target_quat, max_steps=100, threshold=0.02, timeout=10.0)
        if done:
            print("[Task] Terminated during rotate")
            return

        # 2) Move to the side of the drawer
        print("[Task] Moving to side of drawer")
        obs, reward, done = move(
            env,
            task,
            target_pos=side_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during side move")
            return

        # 3) Move to the anchor position for grasping
        print("[Task] Moving to anchor position of drawer handle")
        obs, reward, done = move(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.005,
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during anchor move")
            return

        # 4) Pick (grasp) the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.01,
            max_steps=100,
            threshold=0.005,
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during pick")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(
            env,
            task,
            max_steps=100,
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during pull")
            return

        print("[Task] Drawer has been successfully opened.")

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        # Ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()