import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if needed
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Replace these keys with the actual names in your scene
        # These should correspond to the side and anchor handle of the drawer
        drawer_side_handle_pos = positions['drawer_side_handle']
        drawer_anchor_handle_pos = positions['drawer_anchor_handle']

        # === Oracle Plan Execution ===

        # 1) Move the gripper to the side handle position
        print("[Task] Moving to side handle position:", drawer_side_handle_pos)
        obs, reward, done = move(
            env, task,
            target_pos=drawer_side_handle_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended prematurely during move to side!")
            return

        # 2) Rotate the gripper to the orientation for side approach (ninety_deg)
        # Quaternion for 90 degrees around the Z axis:
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Task] Rotating gripper to ninety degrees around Z-axis")
        obs, reward, done = rotate(
            env, task,
            target_quat=ninety_deg_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended prematurely during rotate!")
            return

        # 3) Move the gripper along the side to the anchor handle position
        print("[Task] Moving to anchor handle position:", drawer_anchor_handle_pos)
        obs, reward, done = move(
            env, task,
            target_pos=drawer_anchor_handle_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended prematurely during move to anchor!")
            return

        # 4) Pick the drawer handle (i.e., grasp the drawer)
        print("[Task] Picking up the drawer handle")
        obs, reward, done = pick(
            env, task,
            target_pos=drawer_anchor_handle_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended prematurely during pick!")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        # Assuming pull signature is (env, task, max_steps, timeout)
        obs, reward, done = pull(
            env, task,
            max_steps=100,
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended prematurely during pull!")
            return

        print("[Task] Drawer should now be open. Plan succeeded!")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()