import numpy as np  
from pyrep.objects.shape import Shape  
from pyrep.objects.proximity_sensor import ProximitySensor  

from env import setup_environment, shutdown_environment  
from skill_code import pick, place, move, rotate, pull  
from video import init_video_writers, recording_step, recording_get_observation  
from object_positions import get_object_positions  

def run_skeleton_task():  
    '''Generic skeleton for running any task in your simulation.'''  
    print("===== Starting Skeleton Task =====")  
    env, task = setup_environment()  
    try:  
        descriptions, obs = task.reset()  
        init_video_writers(obs)  
        task.step = recording_step(task.step)  
        task.get_observation = recording_get_observation(task.get_observation)  

        positions = get_object_positions()  
        # Fetch the needed poses/angles from the dictionary  
        try:  
            drawer_handle_pos = np.array(positions['drawer_handle'])  
            side_pos           = np.array(positions['drawer_side_pos'])  
            anchor_pos         = np.array(positions['drawer_anchor_pos'])  
            zero_angle         = positions['zero_angle']  
            ninety_angle       = positions['ninety_angle']  
        except KeyError as e:  
            print(f"[Error] Missing required position or angle in get_object_positions(): {e}")  
            return  

        # 1) Rotate the gripper from zero to ninety degrees  
        print("[Task] Rotating gripper from", zero_angle, "to", ninety_angle)  
        try:  
            obs, reward, done = rotate(env, task, 'gripper', zero_angle, ninety_angle)  
        except Exception as e:  
            print(f"[Error] rotate failed: {e}")  
            return  
        if done:  
            print("[Task] Task ended during rotate")  
            return  

        # 2) Move gripper to the side position of the drawer  
        current_pos = obs.gripper_pose[:3]  
        print("[Task] Moving to drawer side position:", side_pos)  
        try:  
            obs, reward, done = move(env, task, current_pos, side_pos)  
        except Exception as e:  
            print(f"[Error] move to side_pos failed: {e}")  
            return  
        if done:  
            print("[Task] Task ended during move to side_pos")  
            return  

        # 3) Move gripper to the anchor position on the drawer handle  
        print("[Task] Moving to drawer anchor position:", anchor_pos)  
        try:  
            obs, reward, done = move(env, task, side_pos, anchor_pos)  
        except Exception as e:  
            print(f"[Error] move to anchor_pos failed: {e}")  
            return  
        if done:  
            print("[Task] Task ended during move to anchor_pos")  
            return  

        # 4) Pick up the drawer handle  
        print("[Task] Picking drawer handle at:", anchor_pos)  
        try:  
            obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)  
        except Exception as e:  
            print(f"[Error] pick failed: {e}")  
            return  
        if done:  
            print("[Task] Task ended during pick")  
            return  

        # 5) Pull the drawer open  
        print("[Task] Pulling the drawer...")  
        try:  
            obs, reward, done = pull(env, task)  
        except Exception as e:  
            print(f"[Error] pull failed: {e}")  
            return  
        if done:  
            print("[Task] Task ended after pull")  
            return  

        print("[Task] Drawer should now be open. Task succeeded!")  

    finally:  
        shutdown_environment(env)  

    print("===== End of Skeleton Task =====")  

if __name__ == "__main__":  
    run_skeleton_task()