import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Replace these keys with your actual object names in get_object_positions()
        # For example, if your drawer handle is named 'drawer_handle' in the scene:
        # handle_pos = positions['drawer_handle']
        # drop_zone_pos = positions['drop_zone']
        handle_pos = positions.get('drawer_handle')
        drop_zone_pos = positions.get('drop_zone')

        if handle_pos is None or drop_zone_pos is None:
            raise KeyError("Expected keys 'drawer_handle' and 'drop_zone' in positions")

        # === Oracle Plan Execution ===

        # Step 1: Rotate the gripper to ninety degrees so it can approach the drawer handle
        # Quaternion for a 90° rotation around the gripper's local axis (placeholder)
        ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        print(f"[Task] Rotating gripper to ninety degrees: {ninety_deg_quat}")
        obs, reward, done = rotate(env, task, ninety_deg_quat, max_steps=100, threshold=0.02, timeout=5.0)
        if done:
            print("[Task] Episode ended during rotate")
            return

        # Step 2: Approach and pick the drawer handle
        print(f"[Task] Picking up drawer handle at {handle_pos}")
        obs, reward, done = pick(
            env,
            task,
            target_pos=handle_pos,
            approach_distance=0.10,
            max_steps=150,
            threshold=0.01,
            approach_axis='z',
            timeout=8.0
        )
        if done:
            print("[Task] Episode ended during pick")
            return

        # Step 3: Pull the drawer open by retracting the handle
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Episode ended during pull")
            return

        # Step 4: Place the handle back to its neutral position (optional cleanup)
        print(f"[Task] Placing handle back at drop zone {drop_zone_pos}")
        obs, reward, done = place(
            env,
            task,
            target_pos=drop_zone_pos,
            approach_distance=0.10,
            max_steps=150,
            threshold=0.01,
            timeout=8.0
        )
        if done:
            print("[Task] Episode ended during place")
            return

        print("===== Oracle plan executed successfully =====")

    except Exception as e:
        print(f"[Task] Exception during execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()