import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Task] Retrieved object positions:", positions)

        # === Exploration Phase: Identify objects at each location ===
        # We move the robot to each reported position to trigger any identify/temperature
        # effects and learn missing predicates.
        # We assume move(env, task, target_pos, approach_distance, max_steps, threshold, approach_axis, timeout)
        last_pos = None
        for name, pos in positions.items():
            print(f"[Explore] Moving to {name} at {pos}")
            obs, reward, done = move(
                env, task,
                target_pos=pos,
                approach_distance=0.2,
                max_steps=100,
                threshold=0.02,
                approach_axis='z',
                timeout=5.0
            )
            if done:
                print("[Explore] Task ended prematurely during exploration.")
                return
            last_pos = pos

        # === Oracle Plan Execution ===
        # In this example, we choose two placeholder objects: 'object_1' to pick and
        # 'object_2' as the target placement location.
        # Replace these keys with actual names from get_object_positions() as needed.
        src_key = 'object_1'
        dst_key = 'object_2'
        if src_key not in positions or dst_key not in positions:
            print(f"[Error] Required keys '{src_key}' or '{dst_key}' not found in positions.")
            return

        source_pos = positions[src_key]
        target_pos = positions[dst_key]

        # 1) Pick up object_1
        print(f"[Task] Picking up {src_key} at {source_pos}")
        obs, reward, done = pick(
            env, task,
            target_pos=source_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick!")
            return

        # 2) Optionally rotate the gripper before placement (if orientation matters)
        # Here we demonstrate a rotation to ninety_deg if that skill is relevant.
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Task] Task ended during rotate!")
            return

        # 3) Place the object at the target location
        print(f"[Task] Placing {src_key} at {dst_key} location {target_pos}")
        obs, reward, done = place(
            env, task,
            target_pos=target_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place!")
            return

        # 4) Demonstrate pull action (e.g., opening a drawer handle)
        # If there is an object called 'drawer_handle' in positions, we attempt to pull it.
        handle_key = 'drawer_handle'
        if handle_key in positions:
            handle_pos = positions[handle_key]
            print(f"[Task] Approaching drawer handle at {handle_pos}")
            obs, reward, done = move(
                env, task,
                target_pos=handle_pos,
                approach_distance=0.1,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=5.0
            )
            if done:
                print("[Task] Task ended before pull!")
                return

            print("[Task] Grasping and pulling drawer handle")
            obs, reward, done = pick(
                env, task,
                target_pos=handle_pos,
                approach_distance=0.05,
                max_steps=100,
                threshold=0.005,
                approach_axis='z',
                timeout=5.0
            )
            if done:
                print("[Task] Task ended during drawer pick!")
                return

            obs, reward, done = pull(env, task, handle_key, None)
            if done:
                print("[Task] Task ended after pull!")
                return
        else:
            print(f"[Task] No drawer handle named '{handle_key}' found, skipping pull.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()