import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve known object locations or orientations
        positions = get_object_positions()
        # We expect these keys to exist in the returned dict:
        #   'robot_start'    : robot's start location
        #   'drawer_handle'  : 3‐vector position of drawer handle
        #   'side_pos'       : side approach position for gripper
        #   'anchor_pos'     : anchor approach position for gripper
        #   'drawer_quat'    : quaternion to orient gripper for side approach
        try:
            robot_start = positions['robot_start']
            drawer_handle_pos = positions['drawer_handle']
            side_pos = positions['side_pos']
            anchor_pos = positions['anchor_pos']
            drawer_quat = positions.get('drawer_quat', [0, 0, 0, 1])
        except KeyError as e:
            raise RuntimeError(f"Missing position key: {e}")

        # 1) Rotate gripper to the required orientation for side approach
        print("[Task] Rotating gripper to side‐approach orientation")
        obs, reward, done = rotate(env,
                                   task,
                                   target_quat=drawer_quat,
                                   max_steps=100,
                                   threshold=0.05,
                                   timeout=10.0)
        if done:
            print("[Task] Episode ended during rotate")
            return

        # 2) Move robot from its current location to the side approach position
        print("[Task] Moving to side approach position", side_pos)
        obs, reward, done = move(env,
                                 task,
                                 from_loc=robot_start,
                                 to_loc=side_pos)
        if done:
            print("[Task] Episode ended during move‐to‐side")
            return

        # 3) Move robot from side approach to anchor approach position
        print("[Task] Moving to anchor approach position", anchor_pos)
        obs, reward, done = move(env,
                                 task,
                                 from_loc=side_pos,
                                 to_loc=anchor_pos)
        if done:
            print("[Task] Episode ended during move‐to‐anchor")
            return

        # 4) Pick the drawer handle at the anchor position
        print("[Task] Picking drawer handle at anchor position")
        obs, reward, done = pick(env,
                                 task,
                                 target_pos=drawer_handle_pos,
                                 approach_distance=0.02,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Task] Episode ended during pick‐drawer")
            return

        # 5) Pull the drawer to open it
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env,
                                 task,
                                 max_steps=100,
                                 timeout=10.0)
        if done:
            print("[Task] Episode ended during pull")
            return

        print("[Task] Drawer should now be open")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()