import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example keys in positions dict; replace with actual names if different
        # positions might contain entries like:
        #   'drawer_side', 'drawer_anchor'
        #   'object_1', 'object_2', 'target_zone', etc.
        drawer_side_pos = positions['drawer_side']
        drawer_anchor_pos = positions['drawer_anchor']

        # === Oracle Plan Execution ===

        # 1) Rotate the gripper to align with the drawer handle (90 degrees)
        print("[Task] Rotating gripper to 90° orientation")
        # quaternion for 90° around Z (example)
        ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat,
                                   max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Early termination after rotate")
            return

        # 2) Move the gripper to the side position of the drawer
        print("[Task] Moving gripper to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task,
                                 from_position=None,  # use current internally
                                 to_position=drawer_side_pos)
        if done:
            print("[Task] Early termination after move-to-side")
            return

        # 3) Move the gripper from side to anchor position on the drawer handle
        print("[Task] Moving gripper to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task,
                                 from_position=drawer_side_pos,
                                 to_position=drawer_anchor_pos)
        if done:
            print("[Task] Early termination after move-to-anchor")
            return

        # 4) Grasp the drawer handle
        print("[Task] Picking the drawer handle at anchor position")
        obs, reward, done = pick(env, task,
                                 obj='drawer_handle',
                                 position=drawer_anchor_pos,
                                 gripper='gripper')
        if done:
            print("[Task] Early termination after pick-drawer")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task,
                                 obj='drawer_handle',
                                 gripper='gripper')
        if done:
            print("[Task] Early termination after pull")
            return

        # 6) (Optional) Place any object inside or complete additional sub-tasks here
        #    e.g., place(env, task, obj='some_object', position=target_zone)

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()