import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expecting positions to contain at least:
        # 'drawer_side_pose': [x,y,z,qx,qy,qz,qw]
        # 'drawer_anchor_pose': [x,y,z,qx,qy,qz,qw]
        # 'drawer_open_quat' (optional): [qx,qy,qz,qw]

        drawer_side_pose = positions['drawer_side_pose']
        drawer_anchor_pose = positions['drawer_anchor_pose']
        # If no explicit open quaternion is provided, reuse the anchor orientation
        drawer_open_quat = positions.get('drawer_open_quat', drawer_anchor_pose[3:7])

        # === Oracle Plan Execution ===

        # 1) Move to side position of the drawer
        print("[Task] Moving to side of the drawer at", drawer_side_pose[:3])
        obs, reward, done = move(
            env, task,
            target_pos=drawer_side_pose[:3],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Ended during move-to-side.")
            return

        # 2) Rotate gripper to the opening orientation
        print("[Task] Rotating gripper to open orientation", drawer_open_quat)
        obs, reward, done = rotate(
            env, task,
            target_quat=drawer_open_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Ended during rotate.")
            return

        # 3) Move to anchor position (where the drawer handle is)
        print("[Task] Moving to anchor position at", drawer_anchor_pose[:3])
        obs, reward, done = move(
            env, task,
            target_pos=drawer_anchor_pose[:3],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Ended during move-to-anchor.")
            return

        # 4) Pick (grasp) the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(
            env, task,
            target_pos=drawer_anchor_pose[:3],
            approach_distance=0.02,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Ended during pick.")
            return

        # 5) Pull to open the drawer
        print("[Task] Pulling to open the drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Ended during pull.")
            return

        print("===== Task completed successfully: Drawer opened =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()