import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object and pose information from the environment
        positions = get_object_positions()

        try:
            # Step 1: Rotate gripper to 90 degrees around Z axis
            print("[Plan] Step 1: rotate gripper to 90° orientation")
            # Quaternion for 90° about Z: [x, y, z, w] = [0, 0, sin(45°), cos(45°)]
            target_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Plan] Terminated early after rotate.")
                return

            # Step 2: Move to side position of bottom drawer
            print("[Plan] Step 2: move to side-pos-bottom")
            side_bottom = positions['side-pos-bottom']
            obs, reward, done = move(env, task, side_bottom)
            if done:
                print("[Plan] Terminated early after moving to side-pos-bottom.")
                return

            # Step 3: Move to anchor position of bottom drawer
            print("[Plan] Step 3: move to anchor-pos-bottom")
            anchor_bottom = positions['anchor-pos-bottom']
            obs, reward, done = move(env, task, anchor_bottom)
            if done:
                print("[Plan] Terminated early after moving to anchor-pos-bottom.")
                return

            # Step 4: Pull the drawer open
            print("[Plan] Step 4: pull drawer open")
            # Assuming a pull distance of 0.1m along the local x-axis
            obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
            if done:
                print("[Plan] Terminated early after pull.")
                return

            # Step 5 & 6: pick tomato1 from table and place on plate
            print("[Plan] Step 5: pick tomato1 from table")
            tomato1_pos = positions['tomato1']
            obs, reward, done = pick(env, task, tomato1_pos)
            if done:
                print("[Plan] Terminated early after pick tomato1.")
                return

            print("[Plan] Step 6: place tomato1 on plate")
            plate_pos = positions['plate']
            obs, reward, done = place(env, task, plate_pos)
            if done:
                print("[Plan] Terminated early after place tomato1.")
                return

            # Step 7 & 8: pick tomato2 from table and place on plate
            print("[Plan] Step 7: pick tomato2 from table")
            tomato2_pos = positions['tomato2']
            obs, reward, done = pick(env, task, tomato2_pos)
            if done:
                print("[Plan] Terminated early after pick tomato2.")
                return

            print("[Plan] Step 8: place tomato2 on plate")
            obs, reward, done = place(env, task, plate_pos)
            if done:
                print("[Plan] Terminated early after place tomato2.")
                return

            print("[Task] All steps completed. Goal achieved: drawer open and tomatoes on the plate.")

        except Exception as e:
            print(f"[Error] Exception during plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()