from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np

def run_skeleton_task():
    '''Task: Pull open a drawer then drop two items onto the plate.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get all relevant object positions
        positions = get_object_positions()
        # map our objects
        try:
            drawer_side_pos = np.array(positions['bottom_side_pos'])
            drawer_anchor_pos = np.array(positions['bottom_anchor_pos'])
            item1_pos = np.array(positions['item1'])
            item2_pos = np.array(positions['item2'])
            plate_pos = np.array(positions['plate'])
        except KeyError as e:
            print(f"[Error] Missing object position: {e}")
            return

        # Precompute a 90-degree rotation about Z
        # use the Rotation imported in skill_code
        quat_90 = R.from_euler('z', 90, degrees=True).as_quat()

        # Step 1: Move to drawer side position
        print("[Plan] Step 1: moving to drawer side position")
        obs, reward, done = move(env, task, target_pos=drawer_side_pos)
        if done:
            print("Task ended early at step 1.")
            return

        # Step 2: Rotate gripper to ninety degrees
        print("[Plan] Step 2: rotating gripper to 90°")
        obs, reward, done = rotate(env, task, target_quat=quat_90)
        if done:
            print("Task ended early at step 2.")
            return

        # Step 3: Move to drawer anchor position (handle)
        print("[Plan] Step 3: moving to drawer anchor position")
        obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("Task ended early at step 3.")
            return

        # Step 4: Pull the drawer open
        print("[Plan] Step 4: pulling to open drawer")
        # use a modest pull distance along x axis
        try:
            obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        except Exception as e:
            print(f"[Error] Pull failed: {e}")
            return
        if done:
            print("Task ended early at step 4.")
            return

        # Step 5: Pick up item1
        print("[Plan] Step 5: picking item1")
        obs, reward, done = pick(env, task, target_pos=item1_pos)
        if done:
            print("Task ended early at step 5.")
            return

        # Step 6: Place item1 on the plate
        print("[Plan] Step 6: placing item1 on plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("Task ended early at step 6.")
            return

        # Step 7: Pick up item2
        print("[Plan] Step 7: picking item2")
        obs, reward, done = pick(env, task, target_pos=item2_pos)
        if done:
            print("Task ended early at step 7.")
            return

        # Step 8: Place item2 on the plate
        print("[Plan] Step 8: placing item2 on plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("Task ended early at step 8.")
            return

        print("[Task] Completed all plan steps. Final reward:", reward)

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()