[Frozen Code Start]
    
[Frozen Code End]

import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Define the drawer to open (bottom, middle, or top)
        drawer = 'bottom'
        side_key = f'{drawer}_side_pos'
        anchor_key = f'{drawer}_anchor_pos'
        joint_key = f'{drawer}_joint_pos'
        side_pos = np.array(positions[side_key])
        anchor_pos = np.array(positions[anchor_key])
        joint_pos = np.array(positions[joint_key])
        # Define plate and items
        plate_pos = np.array(positions['plate'])
        items = ['item1', 'item2']  # tomatoes
        item_positions = [np.array(positions[item]) for item in items]

        # === Oracle Plan Execution ===
        # Step 1: rotate gripper to 90 degrees about Y axis
        print("[Plan] Step 1: rotate to ninety_deg")
        target_quat = R.from_euler('y', 90, degrees=True).as_quat()  # x,y,z,w format
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Plan] Early termination after rotate.")
            return

        # Step 2: move to drawer side position
        print(f"[Plan] Step 2: move to side position of {drawer} drawer at {side_pos}")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Plan] Early termination after first move.")
            return

        # Step 3: move to drawer anchor position
        print(f"[Plan] Step 3: move to anchor position of {drawer} drawer at {anchor_pos}")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Early termination after second move.")
            return

        # Step 4: pull drawer fully open
        print(f"[Plan] Step 4: pull {drawer} drawer open")
        pull_distance = np.linalg.norm(anchor_pos - joint_pos)
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Plan] Early termination after pull.")
            return

        # Steps 5 & 6: pick tomatoes
        for idx, item_pos in enumerate(item_positions, start=5):
            print(f"[Plan] Step {idx}: pick item at {item_pos}")
            obs, reward, done = pick(
                env,
                task,
                target_pos=item_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Plan] Early termination after pick {idx}.")
                return

        # Steps 7 & 8: place tomatoes on the plate
        for idx_incr, item_pos in enumerate(item_positions, start=7):
            print(f"[Plan] Step {idx_incr}: place item on plate at {plate_pos}")
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Plan] Early termination after place {idx_incr}.")
                return

        # Final check
        print("[Task] Completed oracle plan. Reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during plan execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
