from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running the combined-domain task.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Extract positions for required objects
        side_pos_bottom = positions['side-pos-bottom']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']

        # Precompute quaternions for rotations
        angle_to_quat = {
            'zero_deg': R.from_euler('z', 0, degrees=True).as_quat(),
            'ninety_deg': R.from_euler('z', 90, degrees=True).as_quat()
        }

        # === Execute Oracle Plan ===
        # Step 1: move to side-pos-bottom
        print("[Plan] Step 1: move to side-pos-bottom at", side_pos_bottom)
        obs, reward, done = move(env, task, target_pos=side_pos_bottom)
        if done:
            print("[Plan] Task ended unexpectedly after move.")
            return

        # Step 2: rotate from zero_deg to ninety_deg
        print("[Plan] Step 2: rotate to ninety_deg")
        target_quat = angle_to_quat['ninety_deg']
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Plan] Task ended unexpectedly after rotate.")
            return

        # Step 3: pull drawer open
        print("[Plan] Step 3: pull drawer open")
        # Assumed pull distance and axis for drawer
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Task ended unexpectedly after pull.")
            return

        # Step 4: pick tomato1 from table
        print("[Plan] Step 4: pick tomato1 at", tomato1_pos)
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Plan] Task ended unexpectedly after picking tomato1.")
            return

        # Step 5: place tomato1 on plate
        print("[Plan] Step 5: place tomato1 at", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Plan] Task ended unexpectedly after placing tomato1.")
            return

        # Step 6: pick tomato2 from table
        print("[Plan] Step 6: pick tomato2 at", tomato2_pos)
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Plan] Task ended unexpectedly after picking tomato2.")
            return

        # Step 7: place tomato2 on plate
        print("[Plan] Step 7: place tomato2 at", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Plan] Task ended unexpectedly after placing tomato2.")
            return

        # Check final success
        print("[Plan] Completed all steps. Final reward:", reward, "Done flag:", done)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()