[Frozen Code Start]
# run_skeleton_task.py (Fully Generic Skeleton)
#
# Description:
#  - This code is a skeleton that shows how to set up an environment, retrieve
#    object positions, and execute a plan without being tied to a specific skill
#    like pick or place.
#  - Use this as a base for creating your own task logic.
#  - The sections that are task-dependent (e.g., manipulation steps) are marked
#    with TODO or placeholders.

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # you don't need to redefine primitives like move, pick, place

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # TODO: Continue adding more plan steps as needed...

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
[Frozen Code End]

# Code After Frozen Code End

def run_skeleton_task():
    '''Task-specific implementation for pulling a drawer and placing two tomatoes.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Keys we expect: 'bottom_side_pos', 'bottom_anchor_pos', 'item1', 'item2', 'plate'
        bottom_side = positions['bottom_side_pos']
        bottom_anchor = positions['bottom_anchor_pos']
        item1_pos = positions['item1']
        item2_pos = positions['item2']
        plate_pos = positions['plate']

        # Step 1: Approach the drawer side position
        print("[Task] Moving to bottom drawer side position:", bottom_side)
        obs, reward, done = move(env, task, bottom_side)
        if done:
            print("[Task] Task ended prematurely after moving to side position!")
            return

        # Step 2: Move to the drawer anchor position for pulling
        print("[Task] Moving to bottom drawer anchor position:", bottom_anchor)
        obs, reward, done = move(env, task, bottom_anchor)
        if done:
            print("[Task] Task ended prematurely after moving to anchor position!")
            return

        # Step 3: Pull to open the drawer
        print("[Task] Pulling to open bottom drawer")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Task ended prematurely during pull!")
            return

        # Step 4: Pick the first tomato (item1)
        print("[Task] Picking item1 at position:", item1_pos)
        obs, reward, done = pick(env, task, item1_pos)
        if done:
            print("[Task] Task ended prematurely during pick(item1)!")
            return

        # Step 5: Place the first tomato onto the plate
        print("[Task] Placing item1 on the plate at:", plate_pos)
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Task ended prematurely during place(item1)!")
            return

        # Step 6: Pick the second tomato (item2)
        print("[Task] Picking item2 at position:", item2_pos)
        obs, reward, done = pick(env, task, item2_pos)
        if done:
            print("[Task] Task ended prematurely during pick(item2)!")
            return

        # Step 7: Place the second tomato onto the plate
        print("[Task] Placing item2 on the plate at:", plate_pos)
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Task ended prematurely during place(item2)!")
            return

        # Final status
        if done or reward is not None:
            print("[Task] Plan executed. Final reward:", reward)
        else:
            print("[Task] Plan executed. done flag is", done)

    except Exception as e:
        print("[Task] Exception during execution:", e)
        raise
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()