[Frozen Code Start]
    
[Frozen Code End]

import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Execute the oracle plan: open middle drawer and move tomatoes to plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap task step and observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Convert to numpy arrays for skill calls
        def arr(key):
            pos = positions.get(key)
            return np.array(pos, dtype=np.float32) if pos is not None else None

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Step 1: Rotating gripper to 90 degrees")
        # Quaternion for 90° around z-axis: [x, y, z, w]
        target_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)], dtype=np.float32)
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Finished early after rotate")
            return

        # Step 2: Move to side position of middle drawer
        print("[Task] Step 2: Moving to side-pos-middle")
        side_middle = arr('side-pos-middle')
        if side_middle is None:
            raise KeyError("Position for side-pos-middle not found")
        obs, reward, done = move(env, task, target_pos=side_middle)
        if done:
            print("[Task] Finished early after move-to-side")
            return

        # Step 3: Move to anchor position of middle drawer
        print("[Task] Step 3: Moving to anchor-pos-middle")
        anchor_middle = arr('anchor-pos-middle')
        if anchor_middle is None:
            raise KeyError("Position for anchor-pos-middle not found")
        obs, reward, done = move(env, task, target_pos=anchor_middle)
        if done:
            print("[Task] Finished early after move-to-anchor")
            return

        # Step 4: Pick (grasp) the drawer handle
        print("[Task] Step 4: Grasping drawer handle (pick-drawer)")
        obs, reward, done = pick(env, task, target_pos=anchor_middle)
        if done:
            print("[Task] Finished early after pick-drawer")
            return

        # Step 5: Pull drawer open
        print("[Task] Step 5: Pulling drawer open")
        # Pull distance and axis may need tuning; assume x-axis inward pull
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')
        if done:
            print("[Task] Finished early after pull")
            return

        # Step 6: Pick tomato1 from table
        print("[Task] Step 6: Picking tomato1")
        tomato1_pos = arr('tomato1')
        if tomato1_pos is None:
            raise KeyError("Position for tomato1 not found")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Finished early after picking tomato1")
            return

        # Step 7: Place tomato1 on plate
        print("[Task] Step 7: Placing tomato1 on plate")
        plate_pos = arr('plate')
        if plate_pos is None:
            raise KeyError("Position for plate not found")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Finished early after placing tomato1")
            return

        # Step 8: Pick tomato2 from table
        print("[Task] Step 8: Picking tomato2")
        tomato2_pos = arr('tomato2')
        if tomato2_pos is None:
            raise KeyError("Position for tomato2 not found")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Finished early after picking tomato2")
            return

        # Step 9: Place tomato2 on plate
        print("[Task] Step 9: Placing tomato2 on plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Finished early after placing tomato2")
            return

        # Plan complete
        print("[Task] Oracle plan executed. Final reward:", reward)
        if not done:
            print("[Task] Task completed without 'done' flag.")

    finally:
        # Ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()