[Frozen Code Start]
    
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        # We approximate zero_deg as identity quaternion and ninety_deg about z-axis
        zero_quat = np.array([0.0, 0.0, 0.0, 1.0], dtype=np.float32)
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)], dtype=np.float32)
        print("[Plan] Step 1: rotate gripper to ninety_deg")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Plan] Task ended unexpectedly after rotate.")
            return

        # Step 2: Move gripper to side position of bottom drawer
        side_pos = positions['bottom_side_pos']
        print("[Plan] Step 2: move to bottom side pos", side_pos)
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] Task ended unexpectedly during move-to-side.")
            return

        # Step 3: Move gripper to anchor position of bottom drawer
        anchor_pos = positions['bottom_anchor_pos']
        print("[Plan] Step 3: move to bottom anchor pos", anchor_pos)
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Task ended unexpectedly during move-to-anchor.")
            return

        # Step 4: Pick (grab) the drawer handle
        print("[Plan] Step 4: pick drawer handle")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Plan] Task ended unexpectedly during pick-drawer.")
            return

        # Step 5: Pull the drawer open
        # Compute pull distance from anchor to joint position
        joint_pos = positions['bottom_joint_pos']
        pull_distance = np.linalg.norm(joint_pos - anchor_pos)
        print(f"[Plan] Step 5: pull drawer by distance {pull_distance:.3f}")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Plan] Task ended unexpectedly during pull.")
            return

        # Step 6: Pick tomato1 from the table
        tomato1_pos = positions['tomato1']
        print("[Plan] Step 6: pick tomato1 at", tomato1_pos)
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Plan] Task ended unexpectedly during pick tomato1.")
            return

        # Step 7: Place tomato1 onto the plate
        plate_pos = positions['plate']
        print("[Plan] Step 7: place tomato1 on plate at", plate_pos)
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Plan] Task ended after placing tomato1. Reward:", reward)
            return

        # Step 8: Pick tomato2 from the table
        tomato2_pos = positions['tomato2']
        print("[Plan] Step 8: pick tomato2 at", tomato2_pos)
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Plan] Task ended unexpectedly during pick tomato2.")
            return

        # Step 9: Place tomato2 onto the plate
        print("[Plan] Step 9: place tomato2 on plate at", plate_pos)
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Plan] Task completed successfully! Reward:", reward)
        else:
            print("[Plan] Task finished steps but done flag is False. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()

[Frozen Code End]