import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

[Frozen Code Start]

[Frozen Code End]

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Prepare rotation quaternions for zero_deg and ninety_deg ===
        zero_shape = Shape('zero_deg')
        ninety_shape = Shape('ninety_deg')
        zero_quat = normalize_quaternion(zero_shape.get_quaternion())
        ninety_quat = normalize_quaternion(ninety_shape.get_quaternion())

        # === Oracle Plan Execution ===
        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Task ended after rotate! Reward:", reward)
            return

        # Step 2: move-to-side for bottom drawer
        print("[Task] Moving to side position of bottom drawer")
        obs, reward, done = move(env, task, positions['bottom_side_pos'])
        if done:
            print("[Task] Task ended after move-to-side! Reward:", reward)
            return

        # Step 3: move-to-anchor for bottom drawer
        print("[Task] Moving to anchor position of bottom drawer")
        obs, reward, done = move(env, task, positions['bottom_anchor_pos'])
        if done:
            print("[Task] Task ended after move-to-anchor! Reward:", reward)
            return

        # Step 4: pick-drawer (grasp drawer handle)
        print("[Task] Picking drawer handle")
        obs, reward, done = pick(env, task, positions['bottom_joint_pos'])
        if done:
            print("[Task] Task ended after pick-drawer! Reward:", reward)
            return

        # Step 5: pull drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')
        if done:
            print("[Task] Task ended after pull! Reward:", reward)
            return

        # Step 6: pick tomato1 from table
        print("[Task] Picking tomato1 from table")
        obs, reward, done = pick(env, task, positions['tomato1'])
        if done:
            print("[Task] Task ended after picking tomato1! Reward:", reward)
            return

        # Step 7: place tomato1 on plate
        print("[Task] Placing tomato1 on plate")
        obs, reward, done = place(env, task, positions['plate'])
        if done:
            print("[Task] Task completed after placing tomato1! Reward:", reward)
            return

        # Step 8: pick tomato2 from table
        print("[Task] Picking tomato2 from table")
        obs, reward, done = pick(env, task, positions['tomato2'])
        if done:
            print("[Task] Task ended after picking tomato2! Reward:", reward)
            return

        # Step 9: place tomato2 on plate
        print("[Task] Placing tomato2 on plate")
        obs, reward, done = place(env, task, positions['plate'])
        if done:
            print("[Task] Task completed after placing tomato2! Reward:", reward)
        else:
            print("[Task] Task completed! Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()